from jpl_conf import FilePaths
from jpl_conf import Blender_Config_Options
from create_blend import Importer
from create_scene import BuildScene
from osgeo import gdal, osr
from CzmlParser import CZML_Parser
import sys
import math

# Profiling imports to test timing
import time


def timing(f):
    def wrap(*args):
        time1 = time.time()
        ret = f(*args)
        time2 = time.time()
        print('%s function took %0.3f ms' % (f.__name__, (time2 - time1)*1000.0))
    return wrap


def get_meta_data(path):
    bag = gdal.Open(path.get_import_file_name())  # replace it with your file
    # raster is projected
    bag_gtrn = bag.GetGeoTransform()
    bag_proj = bag.GetProjectionRef()
    bag_srs = osr.SpatialReference(bag_proj)
    geo_srs = bag_srs.CloneGeogCS()  # new srs obj to go from x,y -> φ,λ
    transform = osr.CoordinateTransformation(bag_srs, geo_srs)

    bag_bbox_cells = (
        (0., 0.),
        (0, bag.RasterYSize),
        (bag.RasterXSize, bag.RasterYSize),
        (bag.RasterXSize, 0),
    )

    geo_pts = []
    pix_pts = []
    xy_pts = []
    for x, y in bag_bbox_cells:
        x2 = bag_gtrn[0] + bag_gtrn[1] * x + bag_gtrn[2] * y
        y2 = bag_gtrn[3] + bag_gtrn[4] * x + bag_gtrn[5] * y
        geo_pt = transform.TransformPoint(x2, y2)[:2]
        geo_pts.append(geo_pt)
        pix_pts.append([x2, y2])
        xy_pts.append([x, y])

    return [xy_pts, pix_pts, geo_pts]


def generate_sun(sun):
    uv = unitize(sun)
    return zenith(uv), azimuth(uv)


def unitize(v):
    r = math.sqrt((v[0] * v[0]) + (v[1] * v[1]) + (v[2] * v[2]))
    return v[0] / r, v[1] / r, v[2] / r


def azimuth(u):
    return math.atan2(u[1], u[0])


def zenith(u):
    return math.atan2(u[0], u[2])

@timing
def do_import(in_obj, blend_config, user_points):
    in_obj.clear_blend_file()
    objName = "terrain"
    blend_config.set_terrain(objName)

    # Importing Functions.
    # Supports IMG, Collada, and OBJ
    #in_obj.import_hirise_img("BIN2")
    #in_obj.import_hirise_img("BIN6", 0.01)
    in_obj.import_hirise_img("BIN12-FAST", 0.01)
    in_obj.set_material_option()
    in_obj.select_object()

@timing
def do_create_scene(scene):
    scene.camera_path()
    scene.make_camera()
    scene.create_lamp()
    scene.key_frame_camera()
    scene.set_camera_orientation()
    scene.set_cycles_options()

@timing
def do_render(path, split):
    f = open("/home/chrisomlor/MovieDemo/hadoop/input/input.txt",'w')
    frame_count = path.get_frame_count()
    start = end = 1
    job_num = 0
    while end <= frame_count:
        end = start + 59
        #split.create_job(start, end, path.get_abs_path_project(), job_num, 'terrain', path.get_blend_file())
        f.write(str(start) + " " + str(end) + " ")
        if end == frame_count:
            end = 999
        start = end + 1
        job_num += 1
    f.close()

@timing
def do_animate(animater):
    animater.animate()

def main():
    json_parse = CZML_Parser(sys.argv[5])
    point, angle = json_parse.blenderCamera()
    sun_data = json_parse.sundata()

    # Convert Sun Data to usable points
    sun_pos = unitize(sun_data)
    sun_ori = generate_sun(sun_data)

    out_file = 'my_test.blend'
    in_file = 'my_image.IMG'
    text_file = 'texture_sb.jpg'

    # Create Class Objects
    file_path = FilePaths(in_file, out_file, text_file)
    blend_config = Blender_Config_Options()
    meta_data = get_meta_data(file_path)
    my_importer = Importer(file_path, blend_config)
    my_scene = BuildScene(blend_config, file_path, meta_data, [point, angle, sun_ori, sun_pos])

    user_points_converted = []
    for pt in point:
        convert = my_scene.geo_2_pix(float(pt[1]), float(pt[2]), float(pt[3]))
        user_points_converted.append(convert)

    # Execute Class Functions
    do_import(my_importer, blend_config, user_points_converted)
    do_create_scene(my_scene)

    # Save all the options into a blend file
    my_importer.save_scene(out_file)

    do_render(file_path, my_importer)

if __name__ == "__main__":
    main()
