/// A template matching algorithm based on
///
/// Erik R. Urbach, Tomasz F. Stepinski, Automatic detection of sub-km craters in high resolution planetary images,
/// Planetary and Space Science, Volume 57, Issue 7, June 2009, Pages 880-887, ISSN 0032-0633
///
/// Using 2nd and 3rd order invariants described in
///
/// Jan Flusser, On the independence of rotation moment invariants,
/// Pattern Recognition, Volume 33, Issue 9, September 2000, Pages 1405-1410, ISSN 0031-3203
///
/// with weighted euclidean distance as the measure of dissimilarity
///
/// There is also an option to use hu moments or use ordinary euclidean distance
///
/// Department of Computer Science
/// California State University, Los Angeles



package lipfd.templateMatching;

import lipfd.commons.*;

import org.opencv.core.Core;
import org.opencv.core.CvType;
import org.opencv.core.Mat;
import org.opencv.core.Point;
import org.opencv.core.Scalar;
import org.opencv.highgui.*;
import org.opencv.imgproc.Imgproc;

import java.io.IOException;
import java.io.File;
import java.util.*;
import java.util.function.Predicate;
import java.lang.Math;
import java.text.NumberFormat;

import java.lang.InterruptedException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import java.util.concurrent.Callable;



public class TemplateMatching {

	public static int bitDepth = 256;
	public static int connectivity = 4;
	public static double addMarginRatio = 1;
	public static int numInvariants = 6;

	public static List<Crater> run(
		Image img, double sunAzimuth, Integer medianKernelSize,
		double areaFilterSize, double powerFilterThreshold, double shapeFilterMaxDistance,
		Integer shapeFilterMinMatches, double areaRatio, double distanceCoef,
		double maxCombinedElongation, double individualElonCoef,
		double sunAzimuthToleranceCoef, double roiEccentricity,
		String highlightTemplatesFolder, String shadowTemplatesFolder,
		boolean generateCroppedCandidates, boolean generateCroppedShadowImages,
		boolean useWeightedEuclideanDistance, boolean useHuMoments, boolean removeDuplicates,
		boolean verbose){

		////
		////
		//// Run Template Matching
		////
		////
		List<Crater> craters = new ArrayList<Crater>();

		///
		/// Load template images
		///
		if(verbose)
			System.out.print("loading template images: ");
		long startTime = System.nanoTime();
		List<double[]> highlightTemplateInvariants = new ArrayList<double[]>();
		List<double[]> shadowTemplateInvariants = new ArrayList<double[]>();
		double[] highlightWeights = new double[numInvariants];
		double[] shadowWeights = new double[numInvariants];
		processTemplates(highlightTemplatesFolder, highlightTemplateInvariants, highlightWeights, useWeightedEuclideanDistance, useHuMoments);
		processTemplates(shadowTemplatesFolder, shadowTemplateInvariants, shadowWeights, useWeightedEuclideanDistance, useHuMoments);
		if(verbose){
			System.out.println(String.format("%s ms", NumberFormat.getNumberInstance(Locale.US).format((System.nanoTime()-startTime)/1000000)));
			System.out.println("highlight invariant templates");
			displayVectors(highlightTemplateInvariants);
			System.out.println("shadow invariant templates");
			displayVectors(shadowTemplateInvariants);
		}

		///
		/// Step 1: invert the image
		///
		if(verbose)
			System.out.print("applying median filter: ");
		startTime = System.nanoTime();
		//Image img = new Image(inputImage);
		Image inv = img.invert();
		Image highlightsImage = img.medianFilter(medianKernelSize);
		Image shadowsImage = inv.medianFilter(medianKernelSize);
		if(verbose)
			System.out.println(String.format("%s ms", NumberFormat.getNumberInstance(Locale.US).format((System.nanoTime()-startTime)/1000000)));

		///
		/// Generate Max-Tree
		///

		//
		// Initialize Variables
		//
		if(verbose)
			System.out.print("initializing variables: ");
		startTime = System.nanoTime();
		MaxTree highlightsTree = new MaxTree();
		MaxTree shadowsTree = new MaxTree();
		int width = highlightsImage.getWidth();
		int height = highlightsImage.getHeight();
		int size = width * height;
		highlightsTree.status = new int[size];
		shadowsTree.status = new int[size];
		for(int i = 0; i < size; i++){
			highlightsTree.status[i] = MaxTree.ST_NOTANALYZED;
			shadowsTree.status[i] = MaxTree.ST_NOTANALYZED;
		}
		int[] highlightsPixels = null, shadowsPixels = null;
		highlightsPixels = highlightsImage.getPixels();
		shadowsPixels = shadowsImage.getPixels();
		Core.MinMaxLocResult hresult = highlightsImage.minMaxLoc();
		Core.MinMaxLocResult sresult = shadowsImage.minMaxLoc();
		int hMinLevel = (int) hresult.minVal;
		int hMinIndex = ((int) hresult.minLoc.y)  * width + (int) hresult.minLoc.x;
		int sMinLevel = (int) sresult.minVal;
		int sMinIndex = ((int) sresult.minLoc.y)  * width + (int) sresult.minLoc.x;
		ArrayList<LinkedList<Integer>> highlightshqueue = new ArrayList<LinkedList<Integer>>();
		for(int i = 0; i < bitDepth; i++)
			highlightshqueue.add(new LinkedList<Integer>());
		highlightshqueue.get(hMinLevel).add(hMinIndex);
		highlightsTree.status[hMinIndex] = MaxTree.ST_INTHEQUEUE;
		ArrayList<LinkedList<Integer>> shadowshqueue = new ArrayList<LinkedList<Integer>>();
		for(int i = 0; i < bitDepth; i++)
			shadowshqueue.add(new LinkedList<Integer>());
		shadowshqueue.get(sMinLevel).add(sMinIndex);
		shadowsTree.status[sMinIndex] = MaxTree.ST_INTHEQUEUE;
		Node firstHighlightsNode = highlightsTree.nodesAtLevel.get(hMinLevel).get(0);
		firstHighlightsNode.parent = firstHighlightsNode;
		firstHighlightsNode.grayLevel = hMinLevel;
		//firstHighlightsNode.pixels.add(hMinIndex);
		Node firstShadowsNode = shadowsTree.nodesAtLevel.get(hMinLevel).get(0);;
		firstShadowsNode.parent = firstShadowsNode;
		firstShadowsNode.grayLevel = sMinLevel;
		//firstShadowsNode.pixels.add(sMinIndex);
		highlightsTree.status[hMinIndex] = 0;
		shadowsTree.status[sMinIndex] = 0;
		highlightsTree.unfinishedNodeAtLevel[hMinLevel] = true;
		shadowsTree.unfinishedNodeAtLevel[sMinLevel] = true;
		Node[] childNodePtr = new Node[1];
		if(verbose)
			System.out.println(String.format("%s ms", NumberFormat.getNumberInstance(Locale.US).format((System.nanoTime()-startTime)/1000000)));
		if(verbose)
			System.out.print("generating max-trees: ");
		startTime = System.nanoTime();
		childNodePtr[0] = null;
		treeFlood(highlightsTree, highlightshqueue, hMinLevel, width, height, highlightsPixels, childNodePtr);
		childNodePtr[0] = null;
		treeFlood(shadowsTree, shadowshqueue, sMinLevel, width, height, shadowsPixels, childNodePtr);
		if(verbose)
			System.out.println(String.format("%s ms", NumberFormat.getNumberInstance(Locale.US).format((System.nanoTime()-startTime)/1000000)));
		System.gc();

		///
		/// Step 2: Power Filter
		///
		if(verbose)
			System.out.print("applying power filter: ");
		startTime = System.nanoTime();
		attributeFilterDirect(highlightsTree, (node) -> node.power >= powerFilterThreshold);
		attributeFilterDirect(shadowsTree, (node) -> node.power >= powerFilterThreshold);
		if(verbose)
			System.out.println(String.format("%s ms", NumberFormat.getNumberInstance(Locale.US).format((System.nanoTime()-startTime)/1000000)));

		///
		/// Step 3: Area Filter
		///
		if(verbose)
			System.out.print("applying area filter: ");
		startTime = System.nanoTime();
		attributeFilterDirect(highlightsTree, (node) -> node.area >= areaFilterSize);
		attributeFilterDirect(shadowsTree, (node) -> node.area >= areaFilterSize);
		if(verbose)
			System.out.println(String.format("%s ms", NumberFormat.getNumberInstance(Locale.US).format((System.nanoTime()-startTime)/1000000)));

		///
		/// Step 4: Shape Filter
		///
		if(verbose)
			System.out.print("applying shape filter: ");
		startTime = System.nanoTime();
		shapeFilterDirect(highlightsTree, highlightTemplateInvariants, shapeFilterMaxDistance, shapeFilterMinMatches, highlightWeights, useHuMoments);
		shapeFilterDirect(shadowsTree, shadowTemplateInvariants, shapeFilterMaxDistance, shapeFilterMinMatches, shadowWeights, useHuMoments);
		if(verbose)
			System.out.println(String.format("%s ms", NumberFormat.getNumberInstance(Locale.US).format((System.nanoTime()-startTime)/1000000)));

		///
		/// Step 5: Match highlights with shadows
		///
		List<Node> highlightsNodes = highlightsTree.toList((node) -> node.area < img.getArea()/2);
		List<Node> shadowsNodes = shadowsTree.toList((node) -> node.area < img.getArea()/2);

		// double minArea = 100000;
		// for(Node n : highlightsNodes){
		// 	if(n.area < minArea)
		// 		minArea = n.area;
		// }
		// System.out.println(minArea);

		Collections.sort(highlightsNodes, (n1, n2) -> Double.compare(n1.area, n2.area));
		Collections.sort(shadowsNodes, (n1, n2) -> Double.compare(n1.area, n2.area));

		if(verbose){
			System.out.printf("highlights: %d \n" ,highlightsNodes.size());
			System.out.printf("shadows: %d \n", shadowsNodes.size());
		}
		int cores = Runtime.getRuntime().availableProcessors();
		if(verbose)
			System.out.printf("cores: %d\n", cores);
		ExecutorService executor = Executors.newFixedThreadPool(cores);
		List<List<Node>> setsOfHighlights = new ArrayList<List<Node>>();
		List<FutureTask> tasks = new ArrayList<FutureTask>();
		for(int i = 0; i < cores; i++){
			List<Node> highlightsChunk = new ArrayList<Node>();
			for(int j = i * highlightsNodes.size() / cores, k = 0;
				j < (i+1) * highlightsNodes.size() / cores;
				j++, k++)
				highlightsChunk.add(highlightsNodes.get(j));
			setsOfHighlights.add(highlightsChunk);
			tasks.add(new FutureTask<List<Crater>>(
				new SearchForCraters(highlightsChunk,shadowsNodes,sunAzimuth,
								areaRatio,distanceCoef,maxCombinedElongation,
								individualElonCoef,sunAzimuthToleranceCoef,
								roiEccentricity)));
			executor.execute(tasks.get(i));
		}

		if(verbose)
			System.out.print("matching highlights with shadows: ");
		startTime = System.nanoTime();

		try {
			for(int i = 0; i < cores; i++){
				craters.addAll((List<Crater>) tasks.get(i).get());
			}
		} catch (InterruptedException | ExecutionException | CancellationException e) {
			e.printStackTrace();
		}
		if(verbose){
			System.out.println(String.format("%s ms", NumberFormat.getNumberInstance(Locale.US).format((System.nanoTime()-startTime)/1000000)));
			System.out.printf("craters: %d\n", craters.size());
		}
		if (removeDuplicates){
			//
			// Remove Duplicates
			//
			if(verbose)
				System.out.print("removing duplicates: ");
			startTime = System.nanoTime();
			Collections.sort(craters, (c1, c2) -> Double.compare(c1.area, c2.area));
			Collections.sort(craters, (c1, c2) -> Double.compare(c1.centerY, c2.centerY));
			Collections.sort(craters, (c1, c2) -> Double.compare(c1.centerX, c2.centerX));

			int i = 0, j = 0;
			while(true){
				if(i >= craters.size() - 2)
					break;
				j = i;
				while(true){
					if(j >= craters.size() - 1)
						break;
					if(Util.isCloseInOneDimension(craters.get(i), craters.get(i+1))){
						j++;
						if(Util.isSimilar(craters.get(i), craters.get(j))){
							craters.remove(j);
							j--;
						}
					}
					else break;
				}
				i++;
			}
		}
		return craters;
	}

	public static void main(String[] args){
		////
		//// Load necessary libraries
		////

		System.out.print("Loading OpenCV Native Library: ");
		long startTime = System.nanoTime();
		System.loadLibrary(Core.NATIVE_LIBRARY_NAME);
		System.out.println(String.format("%s ms", NumberFormat.getNumberInstance(Locale.US).format((System.nanoTime()-startTime)/1000000)));

		////
		//// Process input arguments
		////
		System.out.print("Processing input arguments: ");
		startTime = System.nanoTime();
		String inputImage = args[0];
		double sunAzimuth = Double.parseDouble(args[1]);
		addMarginRatio = Double.parseDouble(args[2]);
		Integer medianKernelSize = Integer.parseInt(args[3]);
		double areaFilterSize = Double.parseDouble(args[4]);
		double powerFilterThreshold = Double.parseDouble(args[5]);
		double shapeFilterMaxDistance = Double.parseDouble(args[6]);
		Integer shapeFilterMinMatches = Integer.parseInt(args[7]);
		double areaRatio = Double.parseDouble(args[8]);
		double distanceCoef = Double.parseDouble(args[9]);
		double maxCombinedElongation = Double.parseDouble(args[10]);
		double individualElonCoef = Double.parseDouble(args[11]);
		double sunAzimuthToleranceCoef = Double.parseDouble(args[12]);
		double roiEccentricity = Double.parseDouble(args[13]);
		String highlightTemplatesFolder = args[14];
		String shadowTemplatesFolder = args[15];
		boolean generateCroppedCandidates = (args[16].equalsIgnoreCase("yes")?true:false);
		boolean generateCroppedShadowImages = (args[17].equalsIgnoreCase("yes")?true:false);
		boolean useWeightedEuclideanDistance = (args[18].equalsIgnoreCase("yes")?true:false);
		boolean useHuMoments = (args[19].equalsIgnoreCase("yes")?true:false);
		numInvariants = useHuMoments?7:6;
		System.out.println(String.format("%s ms", NumberFormat.getNumberInstance(Locale.US).format((System.nanoTime()-startTime)/1000000)));


		List<Crater> craters = run(
			new Image(inputImage), sunAzimuth, medianKernelSize,
			areaFilterSize, powerFilterThreshold, shapeFilterMaxDistance,
			shapeFilterMinMatches, areaRatio, distanceCoef,
			maxCombinedElongation, individualElonCoef,
			sunAzimuthToleranceCoef, roiEccentricity,
			highlightTemplatesFolder, shadowTemplatesFolder,
			generateCroppedCandidates, generateCroppedShadowImages,
			useWeightedEuclideanDistance, useHuMoments, true, true);



		System.out.println(String.format("%s ms", NumberFormat.getNumberInstance(Locale.US).format((System.nanoTime()-startTime)/1000000)));
		System.out.printf("\ncraters: %d\n", craters.size());

		////
		//// Generate Outputs
		////

		//
		// make sure folders exist
		//
		File mkdirsFile = new File("results/craters/");
		if(generateCroppedShadowImages)
			mkdirsFile = new File("results/craters/shadows/");
		mkdirsFile.mkdirs();


		System.out.print("generating outputs: ");
		startTime = System.nanoTime();

		String metaData = "";

		Image img = new Image(inputImage);
		Mat colorImage = new Mat();
  		Imgproc.cvtColor(img.getMat(), colorImage, Imgproc.COLOR_GRAY2BGR);

  		int i = 0;
		for(Crater crater : craters){
			i++;
			crater.id = i;
			crater.addMargin(img.getWidth(), img.getHeight(), addMarginRatio);
			metaData += crater.serialize() + ";";

			if(generateCroppedCandidates){
				Image croppedCrater = img.crop(crater.enclosingRect[0], crater.enclosingRect[1],
					crater.enclosingRect[2], crater.enclosingRect[3]);
				croppedCrater
					.resize(28, 28)
					.saveImage("results/craters/" + String.valueOf(i) + ".pgm");
			}

			if(generateCroppedShadowImages){
				Image croppedShadow = img.belowLevel(crater.shadowGrayLevel).crop(
				crater.shadowBoundingBox[0], crater.shadowBoundingBox[1], crater.shadowBoundingBox[2],
				crater.shadowBoundingBox[3]);
				croppedShadow.saveImage("results/craters/shadows/" + String.valueOf(i) + ".pgm");
			}


			// Core.rectangle(colorImage, new Point(crater.enclosingRect[0], crater.enclosingRect[1]),
			// 	new Point(crater.enclosingRect[2], crater.enclosingRect[3]), new Scalar(255, 0, 0), 1);

			Core.circle(colorImage, new Point(crater.centerX, crater.centerY), (int) crater.radius,
				new Scalar(255, 0, 0));

			// Core.circle(colorImage, new Point(crater.centerX, crater.centerY),
			// 	2, new Scalar(0,0,255));

			// Core.line(colorImage, new Point(crater.highlightCenterX, crater.highlightCenterY),
			// 	new Point(crater.shadowCenterX, crater.shadowCenterY), new Scalar(0, 0, 255));

			// Core.circle(colorImage, new Point(crater.shadowCenterX, crater.shadowCenterY),
			// 	2, new Scalar(0,0,255));

			// Core.circle(colorImage, new Point(crater.highlightCenterX, crater.highlightCenterY),
			// 	2, new Scalar(0,0,255));

			// Core.putText(colorImage,
			// 	String.format("%d", i),
			// 	new Point(crater.enclosingRect[0] + 1, crater.enclosingRect[1] + 10),
			//  	Core.FONT_HERSHEY_PLAIN, 1, new Scalar(255, 0, 0));

			// Core.putText(colorImage,
			// 	String.format("%d", i),
			// 	new Point((crater.highlightCenterX + crater.shadowCenterX)/2 - 10,
			// 		(crater.highlightCenterY + crater.shadowCenterY)/2),
			//  	Core.FONT_HERSHEY_PLAIN, 1, new Scalar(255, 0, 0));


			//display shadows
			// Core.rectangle(colorImage, new Point(crater.shadowBoundingBox[0], crater.shadowBoundingBox[1]),
			// 	new Point(crater.shadowBoundingBox[2], crater.shadowBoundingBox[3]), new Scalar(0,0,255));
		}

		try{
			Util.writeFile("results/craters/metadata.txt", metaData);
		}
		catch(IOException e){
			e.printStackTrace();
		}
		// displayImage(Mat2BufferedImage(colorImage), "csula craters");
		Image cratersImage = new Image(colorImage);
		cratersImage.saveImage("results/craters/csula-craters.ppm");
		System.out.println(String.format("%s ms", NumberFormat.getNumberInstance(Locale.US).format((System.nanoTime()-startTime)/1000000)));
		System.out.println("\ndone!");
		System.exit(0);
	}



	private static class SearchForCraters implements Callable<List<Crater>> {
		public SearchForCraters(List<Node> highlights,List<Node> shadows,double sunAzimuth,
								double areaRatio,double distanceCoef,double maxCombinedElongation,
								double individualElonCoef,double sunAzimuthToleranceCoef,
								double roiEccentricity){
			this.highlights = highlights; this.shadows = shadows; this.sunAzimuth = sunAzimuth; this.areaRatio = areaRatio;
			this.distanceCoef = distanceCoef; this.maxCombinedElongation = maxCombinedElongation;
			this.individualElonCoef = individualElonCoef; this.sunAzimuthToleranceCoef = sunAzimuthToleranceCoef;
			this.roiEccentricity = roiEccentricity;
		}

		List<Node> highlights;
		List<Node> shadows;
		double sunAzimuth;
		double areaRatio;
		double distanceCoef;
		double maxCombinedElongation;
		double individualElonCoef;
		double sunAzimuthToleranceCoef;
		double roiEccentricity;


		@Override
		public List<Crater> call() throws Exception {
			List<Crater> craters = new ArrayList<Crater>();
			int lowerSIndexBound = 0;

			for(int hIndex = 0; hIndex < highlights.size(); hIndex++){
				for(int sIndex = lowerSIndexBound; sIndex < shadows.size(); sIndex++){
					Node hNode = highlights.get(hIndex);
					Node sNode = shadows.get(sIndex);
					if(sNode.area < areaRatio * hNode.area){
						if(areaRatio * sNode.area > hNode.area){
							if(hNode.distance(sNode) <
								distanceCoef * Math.max(Math.sqrt(hNode.area), Math.sqrt(sNode.area))) {
								double combinedElongation = hNode.combinedElongation(sNode);
								if(combinedElongation < maxCombinedElongation &&
									combinedElongation < individualElonCoef * hNode.elongation &&
									combinedElongation < individualElonCoef * sNode.elongation){
									double angle = hNode.alignmentAngle(sNode);
									boolean angleCriterion = false;
									if(sunAzimuth > Math.PI/2.0 && angle < -Math.PI/2.0)
										angleCriterion = Math.abs(angle - sunAzimuth + 2*Math.PI) < sunAzimuthToleranceCoef * Math.PI;
									else
										angleCriterion = Math.abs(angle - sunAzimuth) < sunAzimuthToleranceCoef * Math.PI;
									if(angleCriterion){
										Crater crater = new Crater();
										hNode.enclosingRectangle(sNode, crater.enclosingRect);
										crater.orientationAngle = angle;
										// crater.highlightNode = hNode;
										// crater.shadowNode = sNode;
										crater.shadowBoundingBox[0] = sNode.minX;
										crater.shadowBoundingBox[1] = sNode.minY;
										crater.shadowBoundingBox[2] = sNode.maxX;
										crater.shadowBoundingBox[3] = sNode.maxY;
										crater.shadowGrayLevel = bitDepth - 1 - sNode.grayLevel;
										crater.highlightCenterX = hNode.centerX;
										crater.highlightCenterY = hNode.centerY;
										crater.shadowCenterX = sNode.centerX;
										crater.shadowCenterY = sNode.centerY;
										double ratio = Math.abs((double)(crater.enclosingRect[3] - crater.enclosingRect[1]) /
											(double)(crater.enclosingRect[2] - crater.enclosingRect[0]));
										if(ratio > 1/roiEccentricity && ratio < roiEccentricity)
											Util.addCrater(craters, crater);
									}
								}
							}
						}
						else {
							lowerSIndexBound = sIndex;
						}
					}
					else break;
				}
			}
			return craters;
		}
	}


	private static class MaxTree {
		public MaxTree(){
			for(int i = 0; i < bitDepth; i++){
				Node node = new Node();
				List<Node> nodes = new ArrayList<Node>();
				nodes.add(node);
				nodesAtLevel.add(nodes);
			}
		}
		public static final int ST_NOTANALYZED = -1;
		public static final int ST_INTHEQUEUE = -2;
		int[] status;
		boolean[] unfinishedNodeAtLevel = new boolean[bitDepth];
		List<List<Node>> nodesAtLevel = new ArrayList<List<Node>>();

		public List<Node> toList(Predicate<Node> p){
			List<Node> flatNodes = new ArrayList<Node>();
			for(List<Node> nodes : nodesAtLevel){
				for(Node node : nodes){
					if(p.test(node))
						flatNodes.add(node);
				}
			}
			return flatNodes;
		}
	}

	private static int treeFlood(MaxTree tree, List<LinkedList<Integer>> hqueue,
		int grayLevel, int width, int height, int[] pixels, Node[] childNodePtr){
		int size = width * height;
		int pixelIndex, parentOfChildLevel;
		Node currentNode = null;
		while(!hqueue.get(grayLevel).isEmpty()){
			pixelIndex = hqueue.get(grayLevel).removeFirst();
			if(currentNode == null){
				currentNode = Util.getLastElement(tree.nodesAtLevel.get(grayLevel));
				currentNode.addToAttributes(pixelIndex % width, pixelIndex / width, pixelIndex);
				currentNode.grayLevel = grayLevel;
				if(childNodePtr[0] != null && childNodePtr[0].grayLevel != -1)
					currentNode.mergeAttributes(childNodePtr[0]);
			} else {
				currentNode.addToAttributes(pixelIndex % width, pixelIndex / width, pixelIndex);
			}
			tree.status[pixelIndex] = 0;
			List<Integer> neighbors = findNeighbors(width, size, pixelIndex);
			for(int neighborIndex : neighbors){
				if(tree.status[neighborIndex] == MaxTree.ST_NOTANALYZED){
					hqueue.get(pixels[neighborIndex]).add(neighborIndex);
					tree.status[neighborIndex] = MaxTree.ST_INTHEQUEUE;
					tree.unfinishedNodeAtLevel[pixels[neighborIndex]] = true;
					if(pixels[neighborIndex] > pixels[pixelIndex]){
						parentOfChildLevel = pixels[neighborIndex];
						childNodePtr[0] = null;
						do{
							parentOfChildLevel = treeFlood(tree, hqueue, parentOfChildLevel,
								width, height, pixels, childNodePtr);
						} while(parentOfChildLevel > grayLevel);
						currentNode.mergeAttributes(childNodePtr[0]);
						currentNode.children.add(childNodePtr[0]);
					}
				}
			}
		}
		//set parents
		tree.nodesAtLevel.get(grayLevel).add(new Node());
		int parentLevel = grayLevel-1;
		while(parentLevel>=0 && tree.unfinishedNodeAtLevel[parentLevel]==false)
			parentLevel--;
		if(parentLevel >= 0){
			currentNode.parent = Util.getLastElement(tree.nodesAtLevel.get(parentLevel));
		}
		else currentNode.parent = currentNode;
		tree.unfinishedNodeAtLevel[grayLevel] = false;
		currentNode.grayLevel = grayLevel;
		currentNode.processAttributes();
		childNodePtr[0] = currentNode;
		return parentLevel;
	}

	private static List<Integer> findNeighbors(int width, int size, int pixelIndex){
		List<Integer> neighbors = new ArrayList<Integer>();
		int x = pixelIndex%width;
		int y = pixelIndex/width;
		if(x<(width - 1))
			neighbors.add(pixelIndex+1);
		if (x>0)
			neighbors.add(pixelIndex-1);
		if(pixelIndex>=width)
			neighbors.add(pixelIndex-width);
		if (pixelIndex + width<size)
			neighbors.add(pixelIndex + width);

		if(connectivity == 8){
			if(y > 0){
				if(x<(width - 1))
					neighbors.add(pixelIndex-width+1);
				if (x>0)
					neighbors.add(pixelIndex-width-1);
			}
			if(y < (size-1)/width){
				if(x<(width - 1))
					neighbors.add(pixelIndex+width+1);
				if (x>0)
					neighbors.add(pixelIndex+width-1);
			}
		}
		return neighbors;
	}

	private static class Node{
		double area = 0;
		double power = 0;
		double centerX = 0;
		double centerY = 0;
		int minX = Integer.MAX_VALUE, minY = Integer.MAX_VALUE, maxX = -1, maxY = -1;
		double elongation = 0;
		int grayLevel = -1;
		int minChildGrayLevel = bitDepth+20000;
		Node parent = null;
		List<Node> children = new ArrayList<Node>();
		double[][] moments = new double[4][4];
		double[] invariants = new double[numInvariants];
		double[] huMoments = new double[7];

		public double distance(Node n){
			return Math.sqrt((double)Math.pow(centerX - n.centerX, 2) + (double)Math.pow(centerY - n.centerY, 2));
		}
		public double combinedElongation(Node n){
			Node ntemp = new Node();
			for(int i = 0; i < 4; i++)
				for(int j = 0; j < 4; j++)
					ntemp.moments[i][j] = moments[i][j] + n.moments[i][j];
			ntemp.processAttributes();
			return ntemp.elongation;
		}
		public double alignmentAngle(Node shadow){
			return Math.atan2(shadow.centerY - centerY, centerX - shadow.centerX);
		}
		public void enclosingRectangle(Node n, int[] coordinates){
			coordinates[0] = Math.min(minX, n.minX);
			coordinates[1] = Math.min(minY, n.minY);
			coordinates[2] = Math.max(maxX, n.maxX);
			coordinates[3] = Math.max(maxY, n.maxY);
		}
		public void addToAttributes(int x, int y, int pixelIndex){
			area++;
			power = area * Math.pow(grayLevel - minChildGrayLevel, 2);
			minX = Math.min(minX, x);
			minY = Math.min(minY, y);
			maxX = Math.max(maxX, x);
			maxY = Math.max(maxY, y);
			double h, v = 1.0;
			for (int q=0; q<4; q++)
		    {
			    h = v;
			    for (int p=0; p<4-q; p++)
			    {
			       	moments[p][q] += h;
			        h *= x;
			    }
			    v *= y;
		    }
		    //processAttributes();
		}

		public void mergeAttributes(Node child){
			for(int i = 0; i < 4; i++)
				for(int j = 0; j < 4; j++)
					moments[i][j] += child.moments[i][j];
			area += child.area;
			if(child.area > 0.9 * area && child.grayLevel < minChildGrayLevel)
				minChildGrayLevel = child.grayLevel;
			power = area * Math.pow(grayLevel - minChildGrayLevel, 2);
			minX = Math.min(minX, child.minX);
			minY = Math.min(minY, child.minY);
			maxX = Math.max(maxX, child.maxX);
			maxY = Math.max(maxY, child.maxY);
			//processAttributes();
		}
		public void reprocessAttributes(){
			minChildGrayLevel = bitDepth + 200;
			for(Node c : children)
				if(c.area > 0.9 * area && c.grayLevel < minChildGrayLevel)
					minChildGrayLevel = c.grayLevel;
			power = area * Math.pow(grayLevel - minChildGrayLevel, 2);
		}
		public void processAttributes(){
			centerX = moments[1][0]/moments[0][0];
		    centerY = moments[0][1]/moments[0][0];

		    double u11 = moments[1][1]-centerX*moments[0][1];
		    double u20 = moments[2][0]-centerX * moments[1][0];
		    double u02 = moments[0][2]-centerY * moments[0][1];
		    double u21 = moments[2][1]-2*centerX*moments[1][1]-centerY*moments[2][0]+2*Math.pow(centerX,2)*moments[0][1];
		    double u12 = moments[1][2]-2*centerY*moments[1][1]-centerX*moments[0][2]+2*Math.pow(centerY,2)*moments[1][0];
		    double u30 = moments[3][0]-3*centerX*moments[2][0]+2*Math.pow(centerX,2)*moments[1][0];
		    double u03 = moments[0][3]-3*centerY*moments[0][2]+2*Math.pow(centerY,2)*moments[0][1];

		    double n11 = u11/Math.pow(moments[0][0],2);
		    double n20 = u20/Math.pow(moments[0][0],2);
		    double n02 = u02/Math.pow(moments[0][0],2);
		    double n21 = u21/Math.sqrt(Math.pow(moments[0][0],5));
		    double n12 = u12/Math.sqrt(Math.pow(moments[0][0],5));
		    double n30 = u30/Math.sqrt(Math.pow(moments[0][0],5));
		    double n03 = u03/Math.sqrt(Math.pow(moments[0][0],5));

		    invariants[0] = n20 + n02;
		    invariants[1] = Math.pow(n30+n12,2)+Math.pow(n21+n03,2);
		    invariants[2] = (n20-n02)*(Math.pow(n30+n12,2)-Math.pow(n21+n03,2))+
	    					4*n11*(n30+n12)*(n21+n03);
	    	invariants[3] = n11*(Math.pow(n30+n12,2)-Math.pow(n03+n21,2))-
	    					(n20-n02)*(n30+n12)*(n03+n21);
	    	invariants[4] = (n30-3*n12)*(n30+n12)*(Math.pow(n30+n12,2)-3*Math.pow(n21+n03,2))+
    						(3*n21-n03)*(n21+n03)*(3*Math.pow(n30+n12,2)-Math.pow(n21+n03,2));
    		invariants[5] = (3*n21-n03)*(n30+n12)*(Math.pow(n30+n12,2)-3*Math.pow(n21+n03,2))-
							(n30-3*n12)*(n21+n03)*(3*Math.pow(n30+n12,2)-Math.pow(n21+n03,2));

			huMoments[0] = invariants[0];
			huMoments[1] = Math.pow(n20 - n02, 2) + 4*Math.pow(n11,2);
			huMoments[2] = Math.pow(n30 - 3*n12, 2) + Math.pow(3*n21 - n03, 2);
			huMoments[3] = invariants[1];
			huMoments[4] = invariants[4];
			huMoments[5] = invariants[2];
			huMoments[6] = invariants[5];

		    elongation = n20 + n02;

		    power = area * Math.pow(grayLevel - minChildGrayLevel, 2);
		}
	}
	private static void processTemplates(String folder, List<double[]> invariants, double[] weights,
		boolean useWeightedEuclideanDistance, boolean useHuMoments){
		List<File> templateFiles = new ArrayList<File>(Arrays.asList((new File(folder)).listFiles()));
		List<Image> templateImages = new ArrayList<Image>();
		for(File f : templateFiles){
			Image im = new Image(f.getPath());
			Node node = new Node();
			node.grayLevel = 255;
			int[] pixels = im.getPixels();
			for(int i = 0; i < pixels.length; i++){
				if(pixels[i] > 127)
					node.addToAttributes(i%im.getWidth(), i/im.getWidth(), i);
			}
			node.processAttributes();
			if(!useHuMoments)
				invariants.add(node.invariants);
			else invariants.add(node.huMoments);
		}
		double sum = 0;
		for(int i = 0; i < numInvariants; i++){
			if (useWeightedEuclideanDistance){
				sum = 0;
				for(double[] inv : invariants){
					sum += Math.pow(inv[i], 2);
				}
				weights[i] = numInvariants/sum;
			}
			else {
				weights[i] = 1;
			}
		}
	}

	private static void attributeFilterDirect(MaxTree tree, Predicate<Node> p){
		for(int l = 0; l < bitDepth; l++){
			List<Node> nodes = tree.nodesAtLevel.get(l);
			for(Iterator<Node> iterator = nodes.iterator(); iterator.hasNext();){
				Node node = iterator.next();
				if(!p.test(node)){
					// node.newLevel = node.parent.newLevel;
					iterator.remove();
					if(node.parent != null && node != node.parent){
						node.parent.children.remove(node);
						for(Node child : node.children){
							child.parent = node.parent;
							node.parent.children.add(child);
						}
						node.parent.reprocessAttributes();
					}
				}
			}
		}
	}
	private static void shapeFilterDirect(MaxTree tree, List<double[]> templates,
		double maxDist, int minMatches, double[] weights, boolean useHuMoments){
		for(int l = 0; l < bitDepth; l++){
			List<Node> nodes = tree.nodesAtLevel.get(l);
			for(Iterator<Node> iterator = nodes.iterator(); iterator.hasNext();){
				Node node = iterator.next();
				if(!isMatch((useHuMoments?node.huMoments:node.invariants), templates, maxDist, minMatches, weights)){
					// node.newLevel = node.parent.newLevel;
					iterator.remove();
					if(node.parent != null && node != node.parent){
						node.parent.children.remove(node);
						for(Node child : node.children){
							child.parent = node.parent;
							node.parent.children.add(child);
						}
					}
				}
			}
		}
	}
	private static double weightedEuclideanDistance(double[] vec1, double[] vec2, double[] weights){
		double distance = 0;
		for(int i = 0; i < vec1.length; i++)
			distance += weights[i] * Math.pow(vec1[i] - vec2[i], 2);
		return Math.sqrt(distance);
	}
	private static boolean isMatch(double[] vec1, List<double[]> vecs, double maxDist, int minMatches, double[] weights){
		int numMatches = 0;
		for(double[] vec : vecs){
			if(weightedEuclideanDistance(vec1, vec, weights) < maxDist){
				numMatches++;
				if(numMatches >= minMatches)
					return true;
			}
		}
		return false;
	}
	private static String vectorToString(double[] vec){
		String result = "";
		for(int i = 0; i < vec.length; i++){
			result += String.format("%.16f, " ,vec[i]);
		}
		return result;
	}
	private static void displayVectors(List<double[]> vecs){
		System.out.println("");
		for(int i = 0; i < vecs.size(); i++){
			System.out.println(String.format("%3d: %s", i+1, vectorToString(vecs.get(i))));
		}
		System.out.println("");
	}
}
