package lipfd.performanceCalculator;

import lipfd.commons.*;

import java.awt.FlowLayout;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferByte;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;

import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.JLabel;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileNameExtensionFilter;


import org.opencv.core.Core;
import org.opencv.core.CvType;
import org.opencv.core.Mat;
import org.opencv.core.Point;
import org.opencv.core.Scalar;
import org.opencv.highgui.*;
import org.opencv.imgproc.Imgproc;

import java.io.IOException;
import java.io.File;
import java.util.*;
import java.lang.Math;

public class PerformanceCalculator {
	public static void main(String[] args){
		System.loadLibrary(Core.NATIVE_LIBRARY_NAME);
		try{
				Image image = new Image(args[0]);
				String groundTruthFileName = args[1];
				String resultFileName = args[2];
				int minx = Integer.parseInt(args[3]);
				int miny = Integer.parseInt(args[4]);
				double addMargin = Double.parseDouble(args[5]);
				double confThreshold = Double.parseDouble(args[6]);
				List<Crater> groundTruth = Util.parseMetaData(Util.readFile(groundTruthFileName),
					minx, miny, minx + image.getWidth(), miny + image.getHeight());
				List<Crater> result = Util.parseMetaData(Util.readFile(resultFileName),
					minx, miny, minx + image.getWidth(), miny + image.getHeight());
				for(Crater c : result)
					c.addMargin(image.getWidth(), image.getHeight(), addMargin);
				List<Crater> tpList = new ArrayList<Crater>();
				List<Crater> fpList = new ArrayList<Crater>();
				for(Crater c : result)
					if(c.conf > confThreshold)
						fpList.add(c);
				List<Crater> fnList = new ArrayList<Crater>();
				Map<String, Double> performance = reportPerformance(groundTruth, result,
					tpList, fpList, fnList, image.getWidth(), image.getHeight(), confThreshold);
				System.out.printf("\nmeasure:                  value:\n");
				System.out.println("-----------------------------------------");
				int i = 0;
				for(Map.Entry pair : performance.entrySet()){
					i++;
					System.out.printf("%-25s %2.2f", pair.getKey(), pair.getValue());
					if(i > 4)
						System.out.println("%");
					else System.out.println("");
				}
				Mat tpImage = new Mat();
				Mat fpImage = new Mat();
				Mat fnImage = new Mat();
				Imgproc.cvtColor(image.getMat(), tpImage, Imgproc.COLOR_GRAY2BGR);
				Imgproc.cvtColor(image.getMat(), fpImage, Imgproc.COLOR_GRAY2BGR);
				Imgproc.cvtColor(image.getMat(), fnImage, Imgproc.COLOR_GRAY2BGR);
				for(Crater crater : tpList){
					Core.rectangle(tpImage, new Point(crater.enclosingRect[0], crater.enclosingRect[1]),
						new Point(crater.enclosingRect[2], crater.enclosingRect[3]), new Scalar(255, 0, 0), 1);
				}
				for(Crater crater : fpList){
					Core.rectangle(fpImage, new Point(crater.enclosingRect[0], crater.enclosingRect[1]),
						new Point(crater.enclosingRect[2], crater.enclosingRect[3]), new Scalar(0, 255, 0), 1);
				}
				for(Crater crater : fnList){
					Core.rectangle(fnImage, new Point(crater.enclosingRect[0], crater.enclosingRect[1]),
						new Point(crater.enclosingRect[2], crater.enclosingRect[3]), new Scalar(0, 0, 255), 1);
				}
				Util.displayImage(Util.Mat2BufferedImage(tpImage), "true positive");
				Util.displayImage(Util.Mat2BufferedImage(fpImage), "false positive");
				Util.displayImage(Util.Mat2BufferedImage(fnImage), "false negative");

		} catch(IOException ex){ex.printStackTrace();}
	}

	public static void run(List<Crater> groundTruth, List<Crater> result, int imageWidth, int imageHeight, double confThreshold){
		List<Crater> tpList = new ArrayList<Crater>();
		List<Crater> fpList = new ArrayList<Crater>(result);
		List<Crater> fnList = new ArrayList<Crater>();
		Map<String, Double> performance = reportPerformance(groundTruth, result,
			tpList, fpList, fnList, imageWidth, imageHeight, confThreshold);
		System.out.printf("\nmeasure:                  value:\n");
		System.out.println("-----------------------------------------");
		int i = 0;
		for(Map.Entry pair : performance.entrySet()){
			i++;
			System.out.printf("%-25s %2.2f", pair.getKey(), pair.getValue());
			if(i > 4)
				System.out.println("%");
			else System.out.println("");
		}
		System.out.println("");
	}

	public static Map<String, Double> reportPerformance(List<Crater> groundTruth, List<Crater> resultCopy,
			List<Crater> tpList, List<Crater> fpList, List<Crater> fnList,
			int imageWidth, int imageHeight, double confThreshold){

		List<Crater> result = new ArrayList<Crater>();
		result.addAll(resultCopy);
		// Collections.sort(groundTruth, (c1, c2) -> Double.compare(c1.centerY, c2.centerY));
		// Collections.sort(groundTruth, (c1, c2) -> Double.compare(c1.centerX, c2.centerX));

		// Collections.sort(result, (c1, c2) -> Double.compare(c1.centerY, c2.centerY));
		// Collections.sort(result, (c1, c2) -> Double.compare(c1.centerX, c2.centerX));
		int resultSize = 0;
		for(int j = 0; j < result.size(); j++)
			if(result.get(j).conf > confThreshold)
				resultSize++;
		double tp = 0, fp = 0, fn = 0;
		double tn = (double)Math.pow(Math.max(imageWidth, imageHeight), 3);
		boolean matchFound = false;
		for(int i = 0; i < groundTruth.size(); i++){
			matchFound = false;
			Crater falseNegative = null;
			for(Iterator<Crater> iterator = result.iterator(); iterator.hasNext();){
				Crater r = iterator.next();
				if(Util.isSimilar_md(groundTruth.get(i), r) &&
					r.conf > confThreshold){
					if(!matchFound)
						tp++;
					else resultSize--;
					// tp++;
					matchFound = true;
					tpList.add(r);
					fpList.remove(r);
					iterator.remove();
					//break;
				}
			}
			if(!matchFound){
				fn++;
				fnList.add(groundTruth.get(i));
			}
		}

		fp = resultSize - tp;

		Map<String, Double> performance = new LinkedHashMap<String, Double>();
		double precision = tp/(tp+fp);
		double recall = tp/(tp + fn);
		double accuracy = (tp+tn)/(tp+fp+fn+tn);
		double f1score = 2 * precision * recall / (precision + recall);
		double falseDiscoveryRate = fp / (fp + tp);
		double falsePositiveRate = fp / (fp + tn);
		double falseNegativeRate = fn / (fn + tp);
		double gscore = Math.sqrt(precision*recall);
		double mcc = (tp * tn - fp * fn) / Math.sqrt((tp + fp)*(tp + fn)*(tn + fp)*(tn + fn));
		performance.put("true positive", tp);
		performance.put("false positive", fp);
		performance.put("false negative", fn);
		performance.put("true negative", tn);
		performance.put("false discovery rate", falseDiscoveryRate * 100);
		performance.put("false positive rate", falsePositiveRate * 100);
		performance.put("false negative rate", falseNegativeRate * 100);
		performance.put("accuracy", accuracy * 100);
		performance.put("precision", precision * 100);
		performance.put("recall", recall * 100);
		performance.put("f1 score", f1score * 100);
		performance.put("g score", gscore * 100);
		performance.put("mcc", mcc * 100);
		return performance;
	}

}
