package lipfd.commons;

import java.lang.Math;
import java.util.*;
import java.util.Map.Entry;
import java.io.*;
import java.net.URL;
import java.net.HttpURLConnection;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferByte;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.JLabel;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileNameExtensionFilter;

import lipfd.commons.model.ImageMetadata;
import lipfd.commons.model.dao.jdbc.ImageMetadataDaoImpl;

import org.opencv.core.Core;
import org.opencv.core.Mat;
import org.opencv.core.Point;
import org.opencv.core.Rect;
import org.opencv.core.Point;
import org.opencv.core.Size;


public class Util{
	public static <T> T getLastElement(List<T> list){
		return list.get(list.size() -1);
	}

	public static double euclideanDistance(double x1, double y1, double x2, double y2){
		return Math.sqrt(Math.pow(x1 - x2, 2) + Math.pow(y1 - y2, 2));
	}

	public static double euclideanDistance(Point p1, Point p2){
		return Math.sqrt(Math.pow(p1.x - p2.x, 2) + Math.pow(p1.y - p2.y, 2));
	}

	public static double distance(Crater c1, Crater c2){
		return euclideanDistance(
			average(c1.enclosingRect[0], c1.enclosingRect[2]),
			average(c1.enclosingRect[1], c1.enclosingRect[3]),
			average(c2.enclosingRect[0], c2.enclosingRect[2]),
			average(c2.enclosingRect[1], c2.enclosingRect[3]));
	}

	public static double average(double a, double b){
		return (a+b)/2.0;
	}
	// The following three functions
	// based on http://www.movable-type.co.uk/scripts/latlong.html
	// halfway-point along a great circle path between two points.
	// all angles in radians.
	public static Point midpoint(double long1, double long2, double lat1, double lat2){
		double bx = Math.cos(lat2) * Math.cos(long2 - long1);
		double by = Math.cos(lat2) * Math.sin(long2 - long1);

		double a = Math.sqrt( Math.pow(Math.cos(lat1)+bx ,2.0) + Math.pow(by, 2.0) );
		double avgLong = Math.atan2(Math.sin(lat1)+Math.sin(lat2), a);
		double avgLat = long1 + Math.atan2(by, Math.cos(lat1)+bx);
		Point midpoint = new Point(avgLong, avgLat);

		return midpoint;
	}
	public static Point subtract(Point p1, Point p2){
		return new Point(p1.x - p2.x, p1.y - p2.y);
	}

	public static double dotProduct(Point v1, Point v2){
		return v1.x * v2.x + v1.y * v2.y;
	}

	public static double haversine(double long1, double long2, double lat1, double lat2){
		return Math.pow(Math.sin((lat2-lat1)/2.0), 2.0) + (Math.cos(lat1)*Math.cos(lat2)*Math.pow((long2-long1)/2.0, 2.0));
	}
	public static double distanceLongLat(double long1, double long2, double lat1, double lat2){
		double moonRadius = 1737500; // in metres

		double a = haversine(long1, long2, lat1, lat2);
		double b = 2.0 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));

		return moonRadius * b;
	}
	public static double[] sphericalToCartesian(double longitude, double latitude){
		double x = Math.sin(latitude)*Math.cos(longitude);
		double y = Math.sin(latitude)*Math.sin(longitude);
		double z = Math.cos(latitude);

		double[] v = {x, y, z};

		return v;
	}
	public static Point cartesianToSpherical(double x, double y, double z){
		double latitude = Math.atan2(z, Math.sqrt(x*x + y*y));
		double longitude = Math.atan2(-y, x);

		Point avgPoint = new Point(longitude, latitude);

		return avgPoint;
	}
	public static double getArea(Crater c){
		return Math.abs((c.enclosingRect[2] - c.enclosingRect[0]) * (c.enclosingRect[3] - c.enclosingRect[1]));
	}

	public static double getAreaUsingLongLat(Crater c){
		return Math.abs((c.enclosingRectLongLat[2] - c.enclosingRectLongLat[0]) * (c.enclosingRectLongLat[3] - c.enclosingRectLongLat[1]));
	}

	public static int makeDivisible(int value, int divisor) {
		int divisibleValue=0;
		int increment=0;

		while(divisibleValue<value)
		{
			divisibleValue=increment*divisor;
			increment++;
		}
		divisibleValue-=divisor;

		return divisibleValue;
	}


	public static boolean isSimilar(Crater c1, Crater c2){
		double area1 = getArea(c1);
		double area2 = getArea(c2);
		if(distance(c1, c2) < 0.7 * Math.sqrt(Math.max(area1, area2)) &&
			Math.min(area1, area2) > 0.3 * Math.max(area1, area2))
			return true;
		return false;
	}

	// based on http://stackoverflow.com/questions/9324339/how-much-do-two-rectangles-overlap
	public static boolean isSimilarOverlapLongLat(Crater c1, Crater c2, double overlap){
		double SI = Math.max(0, Math.min(c1.enclosingRectLongLat[2], c2.enclosingRectLongLat[2]) -
			Math.max(c1.enclosingRectLongLat[0], c2.enclosingRectLongLat[0])) *
			Math.max(0, Math.min(c1.enclosingRectLongLat[3], c2.enclosingRectLongLat[3]) -
						Math.max(c1.enclosingRectLongLat[1], c2.enclosingRectLongLat[1]));
		double SA = Util.getAreaUsingLongLat(c1);
		double SB = Util.getAreaUsingLongLat(c2);
		if(SI/SA > overlap && SI/SB > overlap)
			return true;
		else return false;

	}

	public static boolean isSimilarOverlap(Crater c1, Crater c2, double overlap){
		double SI = Math.max(0, Math.min(c1.enclosingRect[2], c2.enclosingRect[2]) -
			Math.max(c1.enclosingRect[0], c2.enclosingRect[0])) *
			Math.max(0, Math.min(c1.enclosingRect[3], c2.enclosingRect[3]) -
						Math.max(c1.enclosingRect[1], c2.enclosingRect[1]));
		double SA = Util.getArea(c1);
		double SB = Util.getArea(c2);
		if(SI/SA > overlap && SI/SB > overlap)
			return true;
		else return false;

	}

	public static boolean isSimilar_md(Crater c1, Crater c2){
		double area1 = getArea(c1);
		double area2 = getArea(c2);
		// if(distance(c1, c2) < 0.6 * Math.sqrt(Math.max(area1, area2)) &&
		// 	Math.min(area1, area2) / Math.max(area1, area2) > 0.06 + 0.17 * Math.min(Math.pow(Math.min(area1, area2)/40,2), 1)/1)
		// 	return true;
		// return false;

		// if((getArea(c1)+getArea(c2))/2 > 150)
		// 	return isSimilarOverlap(c1, c2, 0.5);
		// else return isSimilarOverlap(c1, c2, 0.25);

		return isSimilarOverlap(c1, c2, Math.min(0.5, 0.5/800.0 * Math.min(area1, area2) + 0.1));
	}
	public static boolean isCloseInOneDimension(Crater c1, Crater c2){
		double criteria = 0.7 * Math.sqrt(Math.max(c1.area, c2.area));
		if(Math.abs(c1.centerX - c2.centerX) < criteria)
			return true;
		if(Math.abs(c1.centerY - c2.centerY) < criteria)
			return true;
		return false;
	}
	public static void addCrater(List<Crater> craters, Crater crater){
		crater.centerX = average(crater.enclosingRect[0], crater.enclosingRect[2]);
		crater.centerY = average(crater.enclosingRect[1], crater.enclosingRect[3]);
		crater.area = getArea(crater);
		crater.radius = Math.max(
			(int)Math.ceil((double)(crater.enclosingRect[2]-crater.enclosingRect[0])/2.0),
			(int)Math.ceil((double)(crater.enclosingRect[3]-crater.enclosingRect[1])/2.0));
		crater.enclosingRect[0] = (int)(crater.centerX - crater.radius);
		crater.enclosingRect[1] = (int)(crater.centerY - crater.radius);
		crater.enclosingRect[2] = (int)(crater.centerX + crater.radius);
		crater.enclosingRect[3] = (int)(crater.centerY + crater.radius);
		craters.add(crater);
	}
	public static String readFile(String fileName) throws IOException {
	    BufferedReader br = new BufferedReader(new FileReader(fileName));
	    try {
	        StringBuilder sb = new StringBuilder();
	        String line = br.readLine();

	        while (line != null) {
	            sb.append(line);
	            sb.append("\n");
	            line = br.readLine();
	        }
	        if(sb.length() > 0)
	        	return sb.substring(0, sb.length()-1).toString();
	        else return sb.toString();
	    } finally {
	        br.close();
	    }
	}
	public static void writeFile(String filename, String content) throws IOException {
		BufferedWriter writer = null;
		File file = new File(filename);
		file.getParentFile().mkdirs();
		try {
			writer = new BufferedWriter(new FileWriter(filename, false));
			writer.write(content);
		} finally {
            try {
                // Close the writer regardless of what happens...
                writer.close();
            } catch (Exception e) {
            }
        }
	}

	public static Boolean fileExists(String filePath){
		File f = new File(filePath);
		if(f.exists() && !f.isDirectory())
			return true;
		else return false;
	}

	/***
	***
	***	Merging crater lists function start
	***
	***/

	public static ArrayList<Crater> mergeCratersPixels(QuadTree<Double, Crater> uniqueCratersQT, List<Crater> cratersList){
		ArrayList<Crater> mergedList = new ArrayList<Crater>();
		Map<Crater, ArrayList<Crater>> matchedCratersMap = new HashMap<Crater, ArrayList<Crater>>();

		for (Crater crater: cratersList)
		{
			Interval<Double> intervalx = new Interval<Double>((double) crater.enclosingRect[0], (double) crater.enclosingRect[2]);
			Interval<Double> intervaly = new Interval<Double>((double) crater.enclosingRect[1], (double) crater.enclosingRect[3]);

			Interval2D<Double> searchRegion = new Interval2D<Double>(intervalx, intervaly);

			ArrayList<Crater> regionCraters = new ArrayList<Crater>();
			uniqueCratersQT.query2D(searchRegion, regionCraters);

			Crater uniqueCrater = null;
			double smallestAreaDiff = Double.POSITIVE_INFINITY;
			for (Crater rc: regionCraters)
			{
				if(rc.id < 0)
					continue;
				if (isSimilarOverlap(crater, rc, 0.50))
				{
					//	If similar find crater with smallest area difference
					double areaDiff = Math.abs(getArea(rc) - getArea(crater));
					if (areaDiff < smallestAreaDiff)
					{
						uniqueCrater = rc;
						smallestAreaDiff = areaDiff;
					}
				}
			}

			if (uniqueCrater==null)
			{
				mergedList.add(crater);
			}
			else
			{
				if (matchedCratersMap.containsKey(uniqueCrater))
				{
					ArrayList<Crater> matchedCraters = matchedCratersMap.get(uniqueCrater);
					matchedCraters.add(crater);
					matchedCratersMap.put(uniqueCrater, matchedCraters);
				}
				else
				{
					ArrayList<Crater> matchedCraters = new ArrayList<Crater>();
					matchedCraters.add(crater);
					matchedCratersMap.put(uniqueCrater, matchedCraters);
				}
			}
		}

		Iterator<Entry<Crater, ArrayList<Crater>>> iter = matchedCratersMap.entrySet().iterator();
        while (iter.hasNext()) {
            Entry<Crater, ArrayList<Crater>> entry = iter.next();
            //Crater uniqueCrater = entry.getKey();
            ArrayList<Crater> matchedCraters = entry.getValue();

            double ulxsum = 0.0;
            double ulysum = 0.0;
            double lrxsum = 0.0;
            double lrysum = 0.0;

            for (Crater mc: matchedCraters)
            {
            	ulxsum += mc.enclosingRect[0];
            	ulysum += mc.enclosingRect[1];
            	lrxsum += mc.enclosingRect[2];
            	lrysum += mc.enclosingRect[3];
            }

            double avgulx = ((double)ulxsum/(double)matchedCraters.size());
            double avguly = ((double)ulysum/(double)matchedCraters.size());
            double avglrx = ((double)lrxsum/(double)matchedCraters.size());
            double avglry = ((double)lrysum/(double)matchedCraters.size());

            double avgcx = (avglrx+avgulx)/2.0;
            double avgcy = (avglry+avguly)/2.0;

            int[] mergedEnclosingRect = {(int)avgulx, (int)avguly, (int)avglrx, (int)avglry};
			Crater mergedCrater = new Crater(mergedEnclosingRect[0], mergedEnclosingRect[1], mergedEnclosingRect[2], mergedEnclosingRect[3], avgcx, avgcy);

			mergedList.add(mergedCrater);
        }

		return mergedList;
	}

	public static ArrayList<Crater> mergeCraters(QuadTree<Double, Crater> uniqueCratersQT, List<Crater> cratersList){
		ArrayList<Crater> mergedList = new ArrayList<Crater>();
		Map<Crater, ArrayList<Crater>> matchedCratersMap = new HashMap<Crater, ArrayList<Crater>>();

		for (Crater crater: cratersList)
		{
			Interval<Double> longInterval = new Interval<Double>(Math.min(crater.enclosingRectLongLat[0], crater.enclosingRectLongLat[2]), Math.max(crater.enclosingRectLongLat[0], crater.enclosingRectLongLat[2]));
			Interval<Double> latInterval = new Interval<Double>(Math.min(crater.enclosingRectLongLat[1], crater.enclosingRectLongLat[3]), Math.max(crater.enclosingRectLongLat[1], crater.enclosingRectLongLat[3]));

			Interval2D<Double> searchRegion = new Interval2D<Double>(longInterval, latInterval);

			ArrayList<Crater> regionCraters = new ArrayList<Crater>();
			uniqueCratersQT.query2D(searchRegion, regionCraters);

			Crater uniqueCrater = null;
			double smallestAreaDiff = Double.POSITIVE_INFINITY;
			for (Crater rc: regionCraters)
			{
				if(rc.id < 0)
					continue;
				if (isSimilarOverlapLongLat(crater, rc, 0.50))
				{
					//	If similar find crater with smallest area difference
					double areaDiff = Math.abs(getArea(rc) - getArea(crater));
					if (areaDiff < smallestAreaDiff)
					{
						uniqueCrater = rc;
						smallestAreaDiff = areaDiff;
					}
				}
			}

			if (uniqueCrater==null)
			{
				mergedList.add(crater);
			}
			else
			{
				if (matchedCratersMap.containsKey(uniqueCrater))
				{
					ArrayList<Crater> matchedCraters = matchedCratersMap.get(uniqueCrater);
					matchedCraters.add(crater);
					matchedCratersMap.put(uniqueCrater, matchedCraters);
				}
				else
				{
					ArrayList<Crater> matchedCraters = new ArrayList<Crater>();
					matchedCraters.add(crater);
					matchedCratersMap.put(uniqueCrater, matchedCraters);
				}
			}
		}

		Iterator<Entry<Crater, ArrayList<Crater>>> iter = matchedCratersMap.entrySet().iterator();
        while (iter.hasNext()) {
            Entry<Crater, ArrayList<Crater>> entry = iter.next();
            //Crater uniqueCrater = entry.getKey();
            ArrayList<Crater> matchedCraters = entry.getValue();

            double widthSum = 0.0;
            double heightSum = 0.0;
            double vxsum = 0.0;
            double vysum = 0.0;
            double vzsum = 0.0;
            for (Crater mc: matchedCraters)
            {
            	double clong = Math.toRadians(mc.centerXLongLat);
            	double clat = Math.toRadians(mc.centerYLongLat);

            	double[] v = sphericalToCartesian(clong, clat);
            	vxsum += v[0];
            	vysum += v[1];
            	vzsum += v[2];

            	double long1 = Math.min(Math.toRadians(mc.enclosingRectLongLat[0]), Math.toRadians(mc.enclosingRectLongLat[2]));
				double long2 = Math.max(Math.toRadians(mc.enclosingRectLongLat[0]), Math.toRadians(mc.enclosingRectLongLat[2]));
				double lat1 = Math.min(Math.toRadians(mc.enclosingRectLongLat[1]), Math.toRadians(mc.enclosingRectLongLat[3]));
				double lat2 = Math.max(Math.toRadians(mc.enclosingRectLongLat[1]), Math.toRadians(mc.enclosingRectLongLat[3]));

				double width = distanceLongLat(long1, long2, lat1, lat1);
				double height = distanceLongLat(long1, long1, lat1, lat2);

				widthSum += width;
				heightSum += height;
            }

            double avgvx = vxsum/(matchedCraters.size()*1.0);
            double avgvy = vysum/(matchedCraters.size()*1.0);
            double avgvz = vzsum/(matchedCraters.size()*1.0);

            Point avgCenter = cartesianToSpherical(avgvx, avgvy, avgvz);

            double avgHalfWidth = widthSum/(matchedCraters.size()*2.0);
            double avgHalfHeight = heightSum/(matchedCraters.size()*2.0);

            double[] mergedEnclosingRectLongLat = {avgCenter.x-avgHalfWidth, avgCenter.y-avgHalfHeight, avgCenter.x+avgHalfWidth, avgCenter.y+avgHalfHeight};
			Crater mergedCrater = new Crater(mergedEnclosingRectLongLat[0], mergedEnclosingRectLongLat[1], mergedEnclosingRectLongLat[2], mergedEnclosingRectLongLat[3], avgCenter.x, avgCenter.y);

			mergedList.add(mergedCrater);
        }

		return mergedList;
	}

	//	This function is for testing purposes only.
//	public static Metadata createMetadata(String filename) throws IOException {
//		Metadata metadata = new Metadata();
//
//		Point upperLeft = new Point(11.06, 24.8);
//		Point lowerLeft = new Point(11.06, 24.24);
//		Point upperRight = new Point(10.98, 24.8);
//		Point subImagePixel = new Point(1016.0, 6875.0);
//		int imageHeight = 18432;
//		int imageWidth = 2532;
//		int subImageHeight = 1500;
//		int subImageWidth = 1500;
//		metadata = new Metadata(upperLeft, lowerLeft, upperRight, subImagePixel, imageHeight, imageWidth, subImageHeight, subImageWidth);
//
//		return metadata;
//	}


	public static ArrayList<Crater> createCraterListFromFile(String filename) throws IOException {
		List<Crater> cratersList = new ArrayList<Crater>();
		BufferedReader br = new BufferedReader(new FileReader(filename));
	    try {
	    	String line = "";
	    	br.readLine();
	    	br.readLine();
	    	br.readLine();

	        while ((line = br.readLine()) != null) {

	        	int centerX = Integer.parseInt(line.split("\\s+")[1].trim());
	        	int centerY = Integer.parseInt(line.split("\\s+")[2].trim());
	        	double radius = Double.parseDouble(line.split("\\s+")[11].trim()) / 2.0;
	        	double majorAxis = Integer.parseInt(line.split("\\s+")[8].trim());
	        	double minorAxis = Integer.parseInt(line.split("\\s+")[9].trim());

	        	String upperLeft = line.split("\\s+")[13].trim();
	        	String lowerRight = line.split("\\s+")[14].trim();

	        	int[] upperLeftCoords = parseCoordinates(upperLeft);
	        	int[] lowerRightCoords = parseCoordinates(lowerRight);

	        	int minx = upperLeftCoords[0];
				int miny = upperLeftCoords[1];
				int maxx = lowerRightCoords[0];
				int maxy = lowerRightCoords[1];

				Crater crater = new Crater(minx, miny, maxx, maxy, centerX, centerY);
				//crater.radius = radius;
				//crater.majorAxis = majorAxis;
				//crater.minorAxis = minorAxis;
				cratersList.add(crater);
	        }
	    } finally {
	        br.close();
	    }
		return (ArrayList<Crater>) cratersList;
	}
	public static int[] parseCoordinates(String str) throws IOException {
		int[] coords = new int[2];

		String strNoParens = str.replaceAll("[()]", "");
    	String[] strCoords = strNoParens.split(",");

    	coords[0] = Integer.parseInt(strCoords[0]);
    	coords[1] = Integer.parseInt(strCoords[1]);

		return coords;
	}
	/***
	***
	***	Merging crater lists function end
	***
	***/


	//using meta data, of image to get full range of x and y coordinates for returning results and getting all craters in a list.
	public static List<Crater> getAllCratersList(QuadTree<Double, Crater> craterTree, String imageName){

		ImageMetadataDaoImpl metadataImp = new ImageMetadataDaoImpl("lipfd", "lipfd");
		ImageMetadata metadata = metadataImp.getMetadata(imageName);

		List<Crater> cratersList = new ArrayList<Crater>();
		//get range for entire tree to search from using metadata files
		double upperLeftLon = metadata.maximum_longitude;
		double upperRightLon = metadata.maximum_longitude;

		double upperLeftLat = metadata.minimum_latitude;
		double lowerLeftLat = metadata.minimum_latitude;


		Interval<Double> lonRange = new Interval<Double>(Math.min(upperLeftLon, upperRightLon), Math.max(upperLeftLon, upperRightLon));
		Interval<Double> latRange = new Interval<Double>(Math.min(upperLeftLat, lowerLeftLat), Math.max(upperLeftLat, lowerLeftLat));
		Interval2D<Double> searchRect = new Interval2D<Double>(lonRange, latRange);

		craterTree.query2D(searchRect, (ArrayList<Crater>)cratersList);

		return cratersList;
	}

	public static List<Crater> parseMetaData(String metaDataString, int minx, int miny,
		int maxx, int maxy){
		List<Crater> craters = new ArrayList<Crater>();
		if(metaDataString.startsWith("%", 0)){
			double resolution = 0.5;
			String[] metadataLines = metaDataString.split("\\r?\\n");
			resolution = Double.parseDouble(metadataLines[0].split(":")[1].trim());
			for(String craterString : metadataLines){
				craterString = craterString.trim();
				if(craterString.substring(0, 1).equals("%"))
					continue;
				String[] attributes = craterString.split("\\s+");
				Crater crater = new Crater();
				crater.centerX = Double.parseDouble(attributes[1]);
				crater.centerY = Double.parseDouble(attributes[2]);
				Double radius = Double.parseDouble(attributes[7]);
				if(crater.centerX > minx && crater.centerY > miny &&
					crater.centerX < maxx && crater.centerY < maxy){
					crater.enclosingRect[0] = (int)Math.round(crater.centerX - radius / resolution - minx);
					crater.enclosingRect[1] = (int)Math.round(crater.centerY - radius / resolution - miny);
					crater.enclosingRect[2] = (int)Math.round(crater.centerX + radius / resolution - minx);
					crater.enclosingRect[3] = (int)Math.round(crater.centerY + radius / resolution - miny);
					double craterWidth = crater.enclosingRect[2] - crater.enclosingRect[0];
					double craterHeight = crater.enclosingRect[3] - crater.enclosingRect[1];
					crater.conf = Double.parseDouble(attributes[9]);
					double insideRatio = 0.7;
					if(((double)(crater.enclosingRect[2]))/craterWidth>insideRatio &&
						((double)(maxx-minx-crater.enclosingRect[0]))/craterWidth>insideRatio &&
						((double)(crater.enclosingRect[3]))/craterHeight>insideRatio &&
						((double)(maxy-miny-crater.enclosingRect[1]))/craterHeight>insideRatio)
						Util.addCrater(craters, crater);
				}
			}
			System.out.printf("loaded %d craters from jpl's metadata file\n", craters.size());
		}
		else {
			String[] cratersString = metaDataString.split(";");
			for(int i = 0; i < cratersString.length; i++){
				if(cratersString[i].isEmpty())
					continue;
				String[] attributes = cratersString[i].split(",");
				Crater crater = new Crater();
				if(attributes.length > 4){
					crater.id = Integer.parseInt(attributes[0]);
					crater.enclosingRect[0] = Integer.parseInt(attributes[1]);
					crater.enclosingRect[1] = Integer.parseInt(attributes[2]);
					crater.enclosingRect[2] = Integer.parseInt(attributes[3]);
					crater.enclosingRect[3] = Integer.parseInt(attributes[4]);
				}
				else continue;
				if(attributes.length > 5){
					crater.highlightCenterX = Double.parseDouble(attributes[5]);
					crater.highlightCenterY = Double.parseDouble(attributes[6]);
					crater.shadowCenterX = Double.parseDouble(attributes[7]);
					crater.shadowCenterY = Double.parseDouble(attributes[8]);
				}
				if(attributes.length > 9){
					crater.conf = Double.parseDouble(attributes[9]);
				}
				double centerx = Util.average(crater.enclosingRect[0], crater.enclosingRect[2]);
				double centery = Util.average(crater.enclosingRect[1], crater.enclosingRect[3]);
				if(centerx > 0 && centery > 0 && centerx < (maxx - minx) && centery < (maxy - miny))
					Util.addCrater(craters, crater);
			}
			System.out.printf("loaded %d craters from the saved metadata file\n", craters.size());
		}
		return craters;
	}
	public static BufferedImage Mat2BufferedImage(Mat m){
		// source: http://answers.opencv.org/question/10344/opencv-java-load-image-to-gui/

	    int type = BufferedImage.TYPE_BYTE_GRAY;
	    if ( m.channels() > 1 ) {
	        type = BufferedImage.TYPE_3BYTE_BGR;
	    }
	    int bufferSize = m.channels()*m.cols()*m.rows();
	    byte [] b = new byte[bufferSize];
	    m.get(0,0,b); // get all the pixels
	    BufferedImage image = new BufferedImage(m.cols(),m.rows(), type);
	    final byte[] targetPixels = ((DataBufferByte) image.getRaster().getDataBuffer()).getData();
	    System.arraycopy(b, 0, targetPixels, 0, b.length);
	    return image;
	}
	public static void displayImage(java.awt.Image img, String title)
   	{
		JFrame frame=new JFrame();
		JScrollPane scrollpane = new JScrollPane();
		//frame.setLayout(new FlowLayout());
		frame.setSize(1200, 700);
		frame.setTitle(title);
		JLabel lbl = new JLabel(new ImageIcon(img));
		scrollpane.getViewport().add(lbl);
		frame.add(scrollpane);
		frame.setVisible(true);
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
   	}

   	public static void saveImage(Mat mat) throws IOException {
		BufferedImage image = Mat2BufferedImage(mat);
		File outFile = new File("result.png");
		try {
			ImageIO.write(image, "png", outFile);
		} catch (IOException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		}
	}

	public static Mat cropImage(Mat mat, int offset, int cols, int rows)
	{
		Mat matCropped;

		Rect roi = new Rect(offset, offset, cols-offset, rows-offset);
		matCropped = new Mat(mat, roi);

		return matCropped;
	}

   	// based on: http://stackoverflow.com/questions/22273045/java-getting-download-progress
   	public static void download(String urlString, String fileName){
   		java.io.BufferedOutputStream bout = null;
   		java.io.BufferedInputStream in = null;
   		try{
   			int bufferSize = 1024 * 1024;
	   		URL url = new URL(urlString);
	   		HttpURLConnection httpConnection = (HttpURLConnection) (url.openConnection());
	   		long completeFileSize = httpConnection.getContentLength();
	   		in = new java.io.BufferedInputStream(httpConnection.getInputStream());
	   		java.io.FileOutputStream fos = new java.io.FileOutputStream(
	                            fileName);
	   		bout = new java.io.BufferedOutputStream(
	                            fos, bufferSize);
	   		byte[] data = new byte[bufferSize];
	        long downloadedFileSize = 0;
	        int x = 0;
	        long startTime = 0;
	        double progress = 0;
	        double speed = 0;
	        int downloadedBytes = 0;
	        long elapsedTime = 0;
	        String speedString = "";
	        startTime = System.nanoTime();
	        while ((x = in.read(data, 0, bufferSize)) >= 0) {
	            downloadedFileSize += x;
	            downloadedBytes += x;
	            elapsedTime = System.nanoTime() - startTime;
	            if(elapsedTime > 1000000000) {
		            progress = (double) downloadedFileSize / (double) completeFileSize;
		            speed = ((double)downloadedBytes) / ((double)(elapsedTime)) * 1000000000;
		            if (speed / 1024 < 1){
		            	speedString = "B/s";
		            }
		            else if(speed / 1024 / 1024 < 1){
		            	speedString = "KB/s";
		            	speed /= 1024;
		            }
		            else if(speed / 1024 / 1024 / 1024 < 1){
		            	speedString = "MB/s";
		            	speed /= 1024*1024;
		            }

		            System.out.printf("\r%d%% receiving at: %1.1f %s %20s", (int)(progress*100), speed, speedString," ");
		            downloadedBytes = 0;
		            elapsedTime = 0;
		            startTime = System.nanoTime();
	        	}
	            bout.write(data, 0, x);
	        }
	        System.out.println("");
    	}
    	catch(FileNotFoundException e){e.printStackTrace();}
    	catch(IOException e){e.printStackTrace();}
    	finally {
    		try {if(bout != null) bout.close();}
    		catch(Exception e){e.printStackTrace();}
    		finally{
    			try {if(in != null) in.close();}
    			catch(Exception e){e.printStackTrace();}
    		}
    	}
   	}

   	// based on: http://www.javaworld.com/article/2071275/core-java/when-runtime-exec---won-t.html?page=2
   	public static void runProgram(String command){
   		try
        {
            Runtime rt = Runtime.getRuntime();
            Process proc = rt.exec(command);
            // any error message?
            StreamGobbler errorGobbler = new
                StreamGobbler(proc.getErrorStream(), "ERROR");

            // any output?
            StreamGobbler outputGobbler = new
                StreamGobbler(proc.getInputStream(), "OUTPUT");

            // kick them off
            errorGobbler.start();
            outputGobbler.start();

            // any error???
            int exitVal = proc.waitFor();
            System.out.println("ExitValue: " + exitVal);
        } catch (Throwable t){
	        t.printStackTrace();
	    }
   	}


	/**************************************************************************
	 * Interpolation Stuff
	 * ************************************************************************/
   	//Methods for interpolation
   	//Converter that lon and lat location given a pixel. This method should take cropped image info for parameters
	//don't feed info from metaData, this methods finds lon and lat of a single pixel from a cropped image.
	public static Point lonLatPixConverter(Point uLeft, Point lRight, Size size, Point pixel){
		double imageWidth = size.width;
		double imageHeight = size.height;

		Point uRight = new Point(lRight.x, uLeft.y);
		Point lLeft = new Point(uLeft.x, lRight.y);

		//Range for width (u-vector)
		double topLatDif = (uRight.y - uLeft.y)/(imageWidth);
		double topLonDif = (uRight.x - uLeft.x)/(imageWidth);

		//range for height (v-vector)
		double sideLatDif = (lLeft.y - uLeft.y)/(imageHeight);
		double sideLonDif = (lLeft.x - uLeft.x)/(imageHeight);
		//pixel local
		double pixCol = pixel.x;
		double pixRow = pixel.y;
		//Calc latitude by adding lats
		double pixelLat = (pixCol * topLonDif) + (pixCol * sideLonDif); // + uLeft.x;
		double pixelLon = (pixRow * topLatDif) + (pixRow * sideLatDif); // + uLeft.y;

		Point lonAndLat = new Point(pixelLat, pixelLon);
		return lonAndLat;
	}




	//This method will return three lat/lon points of a cropped image.
	//Not to be confused with finding pixel lat and lon. This method should take info from meta data file to find
	//corresponding lon/lat points of cropped image corners. Use these three points with method above to find lat/lon of a pixel
	// public static Point[] getSubImagePoints(Point upperLeft, Point upperRight, Point lowerLeft, int fullImageWidth, int fullImageHeight, Point firstSubImagePixel,
	// 										int subImageWidth, int subImageHeight){
	// 	Point[] threeCorners = new Point[3];
	// 	//get two other pixels to calculate
	// 	Point upperRightPixel = new Point(firstSubImagePixel.x + subImageWidth, firstSubImagePixel.y);
	// 	Point lowerLeftPixel = new Point(firstSubImagePixel.x, firstSubImagePixel.y + subImageHeight);
	// 	//Point lowerRightPixel = new Point(firstSubImagePixel.x + subImageWidth, firstSubImagePixel.y + subImageHeight);
	// 	//use interpolation method to find three points, lon and lat
	// 	Point newLowerLeft = lonLatPixConverter(upperLeft, upperRight, lowerLeft, fullImageWidth, fullImageHeight, lowerLeftPixel);
	// 	Point newUpperLeft = lonLatPixConverter(upperLeft, upperRight, lowerLeft, fullImageWidth, fullImageHeight, firstSubImagePixel);
	// 	Point newUpperRight = lonLatPixConverter(upperLeft, upperRight, lowerLeft, fullImageWidth, fullImageHeight, upperRightPixel);
	// 	//Point newLowerRight = lonLatPixConverter(upperLeft, upperRight, lowerLeft, fullImageWidth, fullImageHeight, lowerRightPixel);

	// 	threeCorners[0] = newLowerLeft;
	// 	threeCorners[1] = newUpperLeft;
	// 	threeCorners[2] = newUpperRight;
	// 	//threeCorners[3] = newLowerRight;

	// 	return threeCorners;  //lowerLeft, upperLeft, and upperRight
	// }

	//Given coordinate of a pixel in lon and lat, find pixel x,y coordinates with respect to image. Need lon/lat of image corners and image width/height.
	//New get pixel method.
	public static Point getPixel(ImageMetadata metadata, Point poi){
		Point ulOriginal = new Point(metadata.maximum_longitude, metadata.minimum_latitude);
		Point urOriginal = new Point(metadata.maximum_longitude, metadata.maximum_latitude);
		Point llOriginal = new Point(metadata.minimum_longitude, metadata.minimum_latitude);
		double imageWidth = metadata.line_samples;
		double imageHeight = metadata.image_lines;

		// //pixels using height of image ::VECTOR for Width
		// double topLonDiff = (urOriginal.x - ulOriginal.x)/((double)imageWidth);
		// //double topLatDiff = (urOriginal.y - ulOriginal.y)/(imageWidth * 1.0);

		// //pixels per lat and lon differences over the side of the image  ::VECTOR for Height
		// //double sideLonDiff  = (llOriginal.x - ulOriginal.x)/(imageHeight * 1.0);
		// double sideLatDiff = (llOriginal.y - ulOriginal.y)/((double)imageHeight);

		// double lonPOIDiff = poi.x - ulOriginal.x;
		// double latPOIDiff = poi.y - ulOriginal.y;

		// int pixelXCoord = (int) (lonPOIDiff / topLonDiff);
		// int pixelYCoord = (int) (latPOIDiff / sideLatDiff);

		// return new Point(pixelXCoord, pixelYCoord);

		double ratioX = Math.abs(dotProduct(subtract(poi, ulOriginal), subtract(urOriginal, ulOriginal))) /
			Math.pow(euclideanDistance(urOriginal, ulOriginal),2);

		double ratioY = Math.abs(dotProduct(subtract(poi, ulOriginal), subtract(llOriginal, ulOriginal))) /
			Math.pow(euclideanDistance(ulOriginal, llOriginal),2);

		double pixelXCoord = ratioX * imageWidth;
		double pixelYCoord = ratioY * imageHeight;

 		return new Point(pixelXCoord, pixelYCoord);
	}

	//This method will return the size of a cropped Image given lon, lat and size of original or full sized image.
	public static Size getNewImageDims(ImageMetadata imd, Point newUL, Point newLR){
		// double imageWidth = imd.line_samples;
		// double imageHeight = imd.image_lines;
		// Point originalLR = new Point(imd.lower_right_longitude, imd.lower_right_latitude);
		// Point originalUL = new Point(imd.maximum_longitude, imd.minimum_latitude);

		// double pixelWidth = imageWidth / (originalLR.x - originalUL.x);
		// double pixelHeight = imageHeight / (originalLR.y - originalUL.y);

		// double newWidth = Math.abs( (newLR.x - newUL.x) * pixelWidth );
		// double newHeight = Math.abs( (newLR.y - newUL.y) * pixelHeight );

		// return new Size(newWidth, newHeight);

		Point ur = getPixel(imd, new Point(newLR.x, newUL.y));
		Point ul = getPixel(imd, newUL);
		Point ll = getPixel(imd, new Point(newUL.x, newLR.y));

		return new Size(euclideanDistance(ur, ul), euclideanDistance(ul, ll));
	}

	//returns center ROI in pixel coordinates
	public static Point getCenterofROI(ImageMetadata md, Point ulROI, Point lrROI){

		double roiCenterLon = (lrROI.x + ulROI.x)/2;
		double roiCenterLat = (lrROI.y + ulROI.y)/2;

		Point centerLonLat = new Point(roiCenterLon, roiCenterLat);
		return getPixel(md, centerLonLat);
	}

	//Find the rotation angle between an image and user queryRegion
	public static double getRotationAngle(Point p1, Point p2){
		double rotatedAngle = 0;
		double latDifference = p2.y - p1.y;
		double lonDifference = p2.x - p1.x;
		//double tanInRadians = Math.toRadians(latDifference/lonDifference);
		rotatedAngle = Math.atan2(latDifference, lonDifference) + Math.PI;

		return rotatedAngle;
	}

	/**************************** End Interpolation ******************************************/

	/*************************************************************************
	 *    Check for duplicates and return QuadTree without duplicates       *
	 * **********************************************************************/

	//This funcion will return true or false given a crater, to check if that crater already exists in quad tree.
	public static boolean isDuplicateLongLat(Crater target, QuadTree<Double, Crater> craters){
		boolean duplicate = false;
		boolean check = false;
		ArrayList<Crater> matches = new ArrayList<Crater>();

		//use the crater's bounding box to search in QuadTree
		//int[] targetsRect = target.enclosingRect;//minx miny maxx maxy
		//Point startPoint = new Point(targetsRect[0], targetsRect[1]);
		//Point endPoint = new Point(targetsRect[2], targetsRect[3]);
		double[] targetsRectLongLat = target.enclosingRectLongLat;
		Point startPoint = new Point(targetsRectLongLat[0], targetsRectLongLat[1]);
		Point endPoint = new Point(targetsRectLongLat[2], targetsRectLongLat[3]);

		//create search intervals
		Interval<Double> longRange = new Interval<Double>(Math.min(startPoint.x, endPoint.x), Math.max(startPoint.x, endPoint.x));
		Interval<Double> latRange = new Interval<Double>(Math.min(startPoint.y, endPoint.y), Math.max(startPoint.y, endPoint.y));

		//2D query and return any craters within that range
		craters.query2D(new Interval2D<Double>(longRange, latRange), matches);
		if(matches.size() == 0){
			duplicate = false;
		}else{
			for(Crater crater : matches){
				//check if any craters in the list overlap by over 50%
				check = isSimilarOverlapLongLat(crater, target, 0.5);
				if (check == true){
					duplicate = true;
					//print statements for debugging
//					System.out.println(crater.centerX + " : " + crater.centerY + " region: (" + crater.enclosingRect[0] + ", " + crater.enclosingRect[1] + "), (" +
//					crater.enclosingRect[2] + ", " + crater.enclosingRect[3] + ")");
				}
			}
		}

		return duplicate;
	}

	public static boolean isDuplicatePixels(Crater target, QuadTree<Double, Crater> craters){
		boolean duplicate = false;
		boolean check = false;
		ArrayList<Crater> matches = new ArrayList<Crater>();

		//use the crater's bounding box to search in QuadTree
		int[] targetsRectInPixels = target.enclosingRect;
		Point startPoint = new Point((double)targetsRectInPixels[0], (double)targetsRectInPixels[1]);
		Point endPoint = new Point((double)targetsRectInPixels[2], (double)targetsRectInPixels[3]);

		//create search intervals
		Interval<Double> longRange = new Interval<Double>(Math.min(startPoint.x, endPoint.x), Math.max(startPoint.x, endPoint.x));
		Interval<Double> latRange = new Interval<Double>(Math.min(startPoint.y, endPoint.y), Math.max(startPoint.y, endPoint.y));

		//2D query and return any craters within that range
		craters.query2D(new Interval2D<Double>(longRange, latRange), matches);
		if(matches.size() == 0){
			duplicate = false;
		}else{
			for(Crater crater : matches){
				//check if any craters in the list overlap by over 50%
				check = isSimilarOverlap(crater, target, 0.5);
				if (check == true){
					duplicate = true;
					//print statements for debugging
//					System.out.println(crater.centerX + " : " + crater.centerY + " region: (" + crater.enclosingRect[0] + ", " + crater.enclosingRect[1] + "), (" +
//					crater.enclosingRect[2] + ", " + crater.enclosingRect[3] + ")");
				}
			}
		}
		return duplicate;
	}

	//This method will take in a list of craters, and return a quadTree of UNIQUE craters
	public static QuadTree<Double, Crater> getQuadTreeWithNoDuplicatesLongLat(List<Crater> craterList, double ulLon, double ulLat, double lrLon, double lrLat){
		QuadTree<Double, Crater> craterTree = new QuadTree<Double, Crater>();
		Crater cDummy = new Crater();
		cDummy.id = -5;
		cDummy.enclosingRectLongLat[0] = ulLon;
		cDummy.enclosingRectLongLat[1] = ulLat;
		cDummy.enclosingRectLongLat[2] = lrLon;
		cDummy.enclosingRectLongLat[3] = lrLat;
		craterTree.insert((ulLon + lrLon)/2, (ulLat + lrLat)/2, cDummy);
		int count = 0;
		for(Crater c : craterList){
			//if this crater is not a duplicate
			if(isDuplicateLongLat(c, craterTree) == false){
				//add it to the tree.
				craterTree.insert(c.centerXLongLat, c.centerYLongLat, c);
				//System.out.println("added: (" + c.centerX + ", " + c.centerY + ")");
				count++;
			}
		}
		// System.out.println(count);
		return craterTree;
	}//end getQuadTreeWithNoDuplicates

	//This method will take in a list of craters, and return a quadTree of UNIQUE craters
	public static QuadTree<Double, Crater> getQuadTreeWithNoDuplicatesPixels(List<Crater> craterList, double imageWidth, double imageHeight){
		QuadTree<Double, Crater> craterTree = new QuadTree<Double, Crater>();
		Crater cDummy = new Crater();
		cDummy.id = -5;
		cDummy.enclosingRect[0] = cDummy.enclosingRect[1] = 0;
		cDummy.enclosingRect[2] = (int)imageWidth;
		cDummy.enclosingRect[3] = (int)imageHeight;
		craterTree.insert(imageWidth/2, imageHeight/2, cDummy);
		int count = 0;
		for(Crater c : craterList){
			//if this crater is not a duplicate
			if(isDuplicatePixels(c, craterTree) == false){
				//add it to the tree.
				craterTree.insert(c.centerX, c.centerY, c);
				//System.out.println("added: (" + c.centerX + ", " + c.centerY + ")");
				count++;
			}
		}
		// System.out.println(count);
		return craterTree;
	}//end getQuadTreeWithNoDuplicates

	/****************************** End Duplicates ************************************/

}
