package lipfd.circleHough;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Graphics;
import java.awt.GridLayout;
import java.awt.image.BufferedImage;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;

import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSlider;
import javax.swing.JTabbedPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.opencv.core.Core;
import org.opencv.core.CvType;
import org.opencv.core.Mat;
import org.opencv.core.MatOfDouble;
import org.opencv.core.Point;
import org.opencv.core.Scalar;
import org.opencv.core.Size;
import org.opencv.imgproc.Imgproc;

import lipfd.commons.*;
import lipfd.commons.model.ImageMetadata;
import lipfd.commons.model.dao.ImageMetadataDao;
import lipfd.commons.model.dao.jdbc.ImageMetadataDaoImpl;

public class CircleHoughDetector {

	private final static double[] DELTAS = {0.70, 0.05, 0.05, 0.05, 0.05, 0.05, 0.05, 0.05};
	private final static double[] MIN_DISTANCES = {80.0, 24.0, 16.0, 8.0, 6.0, 6.0, 4.0, 4.0};
	private final static double[] DPS = {2.0, 2.0, 1.0, 2.0, 2.0, 2.0, 2.0, 2.0};
	private final static int[] MIN_RADII = {5, 11, 26, 51, 76, 101, 126, 151};
	private final static int[] MAX_RADII = {10, 25, 50, 75, 100, 125, 150, 175};
	private final static double[] ACCUMS = {20.0, 25.0, 10.0, 40.0, 50.0, 50.0, 50.0, 40.0};
	private final static int RET_KERNEL_SIZE = 3;

	private static Mat matUnprocessed;
	private static double mu = 0.0;
	private static double sigma = 0.0;

	private final Scalar[] colors = {new Scalar(255, 0, 0), new Scalar(0, 255, 0), new Scalar(0, 0, 255), new Scalar(255, 255, 0), new Scalar(255, 0, 255), new Scalar(0, 255, 255), new Scalar(150, 128, 24), new Scalar(25, 145, 76)};
	private final static int HYST_HIGH_MIN=0;
	private final static int HYST_HIGH_MAX=100;

	private static Dimension imgDims;
	private static JFrame frame;
	private static JSlider hystHighSlider;
	private static JPanel threshPanel;
	private static JPanel optionsPanel;
	private static JPanel imagesPanel;
	private static JLabel jlHystHigh;
	private static JScrollPane resultImgPane;
	//private static JScrollPane histImgPane;
	private static JScrollPane retinexImgPane;
	private static JScrollPane medBlurImgPane;
	private static JScrollPane gaussBlurImgPane;
	private static JScrollPane cannyImgPane;
	private static JTabbedPane intermediatePane;

	private static CustomPanel resultImgPanel;
	//private static CustomPanel histImgPanel;
	private static CustomPanel retinexImgPanel;
	private static CustomPanel medBlurImgPanel;
	private static CustomPanel gaussBlurImgPanel;
	private static CustomPanel cannyImgPanel;

	private static boolean runningGUI = false;
	//private static BufferedImage histImg;
	private static BufferedImage retinexImg;
	private static BufferedImage medBlurImg;
	private static BufferedImage gaussBlurImg;
	private static BufferedImage cannyImg;

	public static void main(String[] args) throws IOException {
		System.loadLibrary(Core.NATIVE_LIBRARY_NAME);

		String productID = args[0];
		int ulx = Integer.parseInt(args[1]);
		int uly = Integer.parseInt(args[2]);
		int lrx = Integer.parseInt(args[3]);
		int lry = Integer.parseInt(args[4]);
		double upperHysteresis = Double.parseDouble(args[5]);

		ImageMetadataDao imageDao = new ImageMetadataDaoImpl("lipfd", "lipfd");
		ImageMetadata bestInputImageMetadata = imageDao.getMetadata(productID);
		System.out.println("\nchecking the downloaded files.");
		if(!Util.fileExists("downloadedImages/" + productID + ".IMG")){
			System.out.println("downloading the image:");
			Util.download("http://lroc.sese.asu.edu/data/" + bestInputImageMetadata.file_specification_name,
				"downloadedImages/" + productID + ".IMG");
		}

		String originalInputImagePath = "downloadedImages/" + productID + ".IMG";
		String newInputImagePath = "";
		String commandString = "";
	/*	if(System.getProperty("os.name").equalsIgnoreCase("linux")){
			newInputImagePath = "downloadedImages/" + productID + ".pgm";
			commandString = "gdal_translate -srcwin 0 0 " + String.valueOf(bestInputImageMetadata.line_samples) + " " +
				String.valueOf(bestInputImageMetadata.image_lines) +
				" -ot Byte -of PNM -scale " +
				originalInputImagePath + " " + newInputImagePath;
		}
		else {
			newInputImagePath = "downloadedImages/" + bestInputImageMetadata.product_id + ".tiff";
			commandString = "gdal_translate -srcwin 0 0 " + String.valueOf(bestInputImageMetadata.line_samples) + " " +
				String.valueOf(bestInputImageMetadata.image_lines) + " " + originalInputImagePath +
				" " + newInputImagePath +
				" -ot Byte -scale 0 4095 0 255";
		}
		System.out.println("running gdal:");
		System.out.println(commandString);
		if(!Util.fileExists(newInputImagePath))
			Util.runProgram(commandString);
		*/
		Image inputImage = new Image("downloadedImages/"+bestInputImageMetadata.product_id+".tiff").crop(ulx, uly, lrx, lry);
		Mat mat = inputImage.getMat();

		CircleHoughDetector.runGUI(mat, upperHysteresis);
	}

	public static ArrayList<Crater> run(Mat mat, double hystHigh)
	{
		matUnprocessed = mat.clone();
		Mat matOutput = mat.clone();

		MatOfDouble mean = new MatOfDouble();
		MatOfDouble stddev = new MatOfDouble();
		Core.meanStdDev(matOutput, mean, stddev);
		mu = mean.get(0, 0)[0];
		sigma = stddev.get(0, 0)[0];

		Mat matPreprocessed = preprocessing(mat);

		double upperHystThresh = calculateUpperHystThresh(matPreprocessed, hystHigh);

		edgeDetector(matPreprocessed, upperHystThresh / 2.0, upperHystThresh);

		ArrayList<Crater> accumCircles = houghTransform(matPreprocessed, upperHystThresh);

		//writeData(accumCircles, matOutput.rows(), matOutput.cols());

		return accumCircles;
	}

	public static Mat preprocessing(Mat mat)
	{
		matUnprocessed = mat.clone();
		int rows = matUnprocessed.rows();
		int cols = matUnprocessed.cols();

		Mat matRetinex = matUnprocessed.clone();
		Retinex retinex = new Retinex();
		matRetinex = retinex.SingleScaleRetinex(matRetinex, RET_KERNEL_SIZE);
		retinexImg = Util.Mat2BufferedImage(matRetinex);

		Mat matGrayscale = new Mat(mat.rows(), mat.cols(), mat.type());
		Imgproc.cvtColor(matRetinex, matGrayscale, Imgproc.COLOR_RGB2GRAY);

		Mat matMedBlur =  matGrayscale.clone();
		Imgproc.medianBlur(matGrayscale, matMedBlur, 3);
		medBlurImg = Util.Mat2BufferedImage(matMedBlur);

		Mat matGaussBlur =  matGrayscale.clone();
		Imgproc.GaussianBlur(matMedBlur, matGaussBlur, new Size(5,5), 0);
		gaussBlurImg = Util.Mat2BufferedImage(matMedBlur);

		Imgproc.dilate(matGaussBlur, matGaussBlur, Imgproc.getStructuringElement(Imgproc.MORPH_RECT, new Size(3, 3)));

		return matGaussBlur;
	}

	public static double calculateUpperHystThresh(Mat matPreprocessed, double hystHigh)
	{
		double upperHystThresh;

		Mat dx = new Mat(matPreprocessed.rows(), matPreprocessed.cols(), CvType.CV_32F);
		Mat dy = new Mat(matPreprocessed.rows(), matPreprocessed.cols(), CvType.CV_32F);

		Imgproc.Sobel(matPreprocessed, dx, CvType.CV_32F, 1, 0, 3, 1, 0, Imgproc.BORDER_REPLICATE);
		Imgproc.Sobel(matPreprocessed, dy, CvType.CV_32F, 0, 1, 3, 1, 0, Imgproc.BORDER_REPLICATE);

		Mat magnitudes = new Mat(matPreprocessed.rows(), matPreprocessed.cols(), CvType.CV_32F);

		Core.magnitude(dx, dy, magnitudes);

		double maxMagnitude = Double.NEGATIVE_INFINITY;
		for (int row=0; row<magnitudes.rows(); row++)
		{
			for (int col=0; col<magnitudes.cols(); col++)
			{
				double[] data = magnitudes.get(row, col);
				if (data[0] > maxMagnitude)
				{
					maxMagnitude = data[0];
				}
			}
		}

		upperHystThresh = maxMagnitude * hystHigh;

		return upperHystThresh;
	}

	public static Mat edgeDetector(Mat mat, double hystLow, double hystHigh)
	{
		Mat matCanny =  mat.clone();
		Imgproc.Canny(mat, matCanny, hystLow, hystHigh, 3, false);
		cannyImg = Util.Mat2BufferedImage(matCanny);

		return matCanny;
	}

	public static ArrayList<Crater> houghTransform(Mat mat, double upperHystThresh)
	{
		ArrayList<Crater> accumCircles = new ArrayList<Crater>();

		for (int j=0; j<MIN_RADII.length; j++)
		{
			Mat circles = new Mat();
			Imgproc.HoughCircles(mat, circles, Imgproc.CV_HOUGH_GRADIENT, DPS[j], mat.rows()/MIN_DISTANCES[j], upperHystThresh, ACCUMS[j], MIN_RADII[j], MAX_RADII[j]);

			if (circles.cols() > 0)
			{
				for (int i=0; i<circles.cols(); i++)
				{
					double c[] = circles.get(0,i);

					if (c==null)
						break;

					Point center = new Point(Math.round(c[0]), Math.round(c[1]));
					int radius = (int) Math.round(c[2]);

					int[] enclosingRect = {(int) (center.x-radius), (int) (center.y-radius), (int) (center.x+radius), (int) (center.y+radius)};

					double delta = analyzeRegion(enclosingRect, radius);
					if (delta < DELTAS[j])
					{
						Crater crater = new Crater(enclosingRect[0], enclosingRect[1], enclosingRect[2], enclosingRect[3], (int) center.x, (int) center.y);
						crater.radius = radius;
						accumCircles.add(crater);

					}
				}
			}
		}

		// Search for circles of unknown radius but above 175 pixels
		int maxCraterBound = Math.min(matUnprocessed.rows(), matUnprocessed.cols());
		if (maxCraterBound > 200)
		{
			Mat circles = new Mat();
			Imgproc.HoughCircles(mat, circles, Imgproc.CV_HOUGH_GRADIENT, DPS[DPS.length-1], mat.rows()/MIN_DISTANCES[MIN_DISTANCES.length-1], upperHystThresh, ACCUMS[ACCUMS.length-1], MAX_RADII[MAX_RADII.length-1], maxCraterBound);

			if (circles.cols() > 0)
			{
				for (int i=0; i<circles.cols(); i++)
				{
					double c[] = circles.get(0,i);

					if (c==null)
						break;

					Point center = new Point(Math.round(c[0]), Math.round(c[1]));
					int radius = (int) Math.round(c[2]);

					int[] enclosingRect = {(int) (center.x-radius), (int) (center.y-radius), (int) (center.x+radius), (int) (center.y+radius)};

					double delta = analyzeRegion(enclosingRect, radius);
					if (delta < DELTAS[DELTAS.length-1])
					{
						Crater crater = new Crater(enclosingRect[0], enclosingRect[1], enclosingRect[2], enclosingRect[3], (int) center.x, (int) center.y);
						crater.radius = radius;
						accumCircles.add(crater);
					}
				}
			}
		}

		return accumCircles;
	}

	public static double analyzeRegion(int[] enclosingRect, int radius)
	{
		double delta = 0.0;

		if ( (enclosingRect[0] < 0) || (enclosingRect[2] > matUnprocessed.cols()) )
		{
			return 1.0;
		}
		if( (enclosingRect[1] < 0) || (enclosingRect[3] > matUnprocessed.rows()) )
		{
			return 1.0;
		}

		int[] bins = new int[256];
		for (int row=enclosingRect[0]; row<enclosingRect[2]; row++)
		{
			for (int col=enclosingRect[1]; col<enclosingRect[3]; col++)
			{
				int data = (int) matUnprocessed.get(col, row)[0];
				bins[data] += 1;
			}
		}

		double mean = 0.0;
		double total = 0.0;
		for (int i=0; i<bins.length; i++)
		{
			mean += i*bins[i];
			total += bins[i];
		}
		mean /= total;

		double stddev = 0.0;
		double sqrdevs = 0.0;
		for (int i=0; i<bins.length; i++)
		{
			sqrdevs += Math.pow(i-mean, 2.0)*bins[i];
		}
		sqrdevs /= total;
		stddev = Math.sqrt(sqrdevs);

		double low = 0.0;
		double high = 0.0;

		if (stddev > mean)
		{
			low = mu - sigma;
			high = mu + sigma;
		}
		else
		{
			low = mean-stddev;
			high = mean+stddev;
		}

		int darkPixels = 0;
		int lightPixels = 0;


		for (int i=0; i<bins.length; i++)
		{
			if (i<low)
			{
				darkPixels += bins[i];
			}
			if (i>high)
			{
				lightPixels += bins[i];
			}
		}

		double avgLightDark = (darkPixels + lightPixels) / 2.0;
		delta = Math.abs(lightPixels - darkPixels) / avgLightDark;

		return delta;
	}

	public static void runGUI(Mat mat, double hystHigh)
	{
		runningGUI = true;
		ArrayList<Crater> cratersList = run(mat, hystHigh);

		Mat matResult = matUnprocessed.clone();
		for(int i=0; i<cratersList.size(); i++)
		{
			Crater crater = cratersList.get(i);
			int[] enclosingRect = crater.enclosingRect;
			Point upperLeftCorner = new Point(enclosingRect[0], enclosingRect[1]);
			Point lowerRightCorner = new Point(enclosingRect[2], enclosingRect[3]);
			Core.rectangle(matResult, upperLeftCorner, lowerRightCorner, new Scalar(255,0,0));
		}

		BufferedImage resultImg = Util.Mat2BufferedImage(matResult);

		frame = new JFrame();
		imgDims = new Dimension(matResult.width(), matResult.height());

		// Canny Edge Detector section
		threshPanel = new JPanel();
		threshPanel.setBorder(BorderFactory.createTitledBorder("Hysteresis Thresholds"));
		threshPanel.setLayout(new GridLayout(2, 2));

		hystHighSlider = new JSlider();
		hystHighSlider.setValue(9);
		hystHighSlider.setMinimum(HYST_HIGH_MIN);
		hystHighSlider.setMaximum(HYST_HIGH_MAX);
		hystHighSlider.setMinorTickSpacing(1);
		hystHighSlider.setMajorTickSpacing(5);
		hystHighSlider.setPaintTicks(true);
		hystHighSlider.setPaintLabels(true);

		jlHystHigh = new JLabel("Upper Hysteresis Threshold:");

		threshPanel.add(jlHystHigh);
		threshPanel.add(hystHighSlider);

		// Combine Panels
		optionsPanel = new JPanel();
		optionsPanel.setLayout(new GridLayout(3,0));
		optionsPanel.add(threshPanel);

		hystHighSlider.addChangeListener(new ChangeListener()
		{
			@Override
			public void stateChanged(ChangeEvent ce)
			{
				JSlider js = (JSlider) ce.getSource();

				if (!js.getValueIsAdjusting())
				{
					double userHystHigh = js.getValue();
					if (userHystHigh == 0.0)
					{
						userHystHigh = 1.0;
					}
					userHystHigh /= 100.0;
					//System.out.println("Hyst. high: "+hystHigh);

					updateGUI(matUnprocessed, userHystHigh);
				}
			}
		});

		// Images Panel
		imagesPanel = new JPanel();
		imagesPanel.setLayout(new GridLayout(0,2));

		/*
		histImgPanel = new CustomPanel(histImg);
		histImgPanel.setPreferredSize(imgDims);
		histImgPane = new JScrollPane(histImgPanel);
		*/

		retinexImgPanel = new CustomPanel(retinexImg);
		retinexImgPanel.setPreferredSize(imgDims);
		retinexImgPane = new JScrollPane(retinexImgPanel);

		medBlurImgPanel = new CustomPanel(medBlurImg);
		medBlurImgPanel.setPreferredSize(imgDims);
		medBlurImgPane = new JScrollPane(medBlurImgPanel);

		gaussBlurImgPanel = new CustomPanel(gaussBlurImg);
		gaussBlurImgPanel.setPreferredSize(imgDims);
		gaussBlurImgPane = new JScrollPane(gaussBlurImgPanel);

		cannyImgPanel = new CustomPanel(cannyImg);
		cannyImgPanel.setPreferredSize(imgDims);
		cannyImgPane = new JScrollPane(cannyImgPanel);

		resultImgPanel = new CustomPanel(resultImg);
		resultImgPanel.setLayout(new FlowLayout());
		resultImgPanel.setPreferredSize(imgDims);
		resultImgPane = new JScrollPane(resultImgPanel);
		resultImgPane.setBorder(BorderFactory.createTitledBorder("Result Image"));

		// Tabs
		//intermediateTabPanel = new JPanel();
		intermediatePane = new JTabbedPane();
		intermediatePane.setBorder(BorderFactory.createTitledBorder("Intermediate Steps"));
		intermediatePane.addTab("Retinex", retinexImgPane);
		//intermediatePane.addTab("Histogram", histImgPane);
		intermediatePane.addTab("Medium Blur", medBlurImgPane);
		intermediatePane.addTab("Gaussian Blur", gaussBlurImgPane);
		intermediatePane.addTab("Canny Edge Detector", cannyImgPane);

		imagesPanel.add(resultImgPane);
		imagesPanel.add(intermediatePane);

		frame.add(imagesPanel, BorderLayout.CENTER);
		frame.add(optionsPanel, BorderLayout.SOUTH);

		frame.pack();
		frame.setTitle("Circle Hough Detection");
		frame.setVisible(true);
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
	}

	public static void updateGUI(Mat mat, double hystHigh)
	{
		ArrayList<Crater> cratersList = run(mat, hystHigh);

		retinexImgPanel.setImage(retinexImg);
		medBlurImgPanel.setImage(medBlurImg);
		gaussBlurImgPanel.setImage(gaussBlurImg);
		cannyImgPanel.setImage(cannyImg);
		retinexImgPanel.repaint();
		medBlurImgPanel.repaint();
		gaussBlurImgPanel.repaint();
		cannyImgPanel.repaint();

		Mat matResult = mat.clone();
		for(int i=0; i<cratersList.size(); i++)
		{
			Crater crater = cratersList.get(i);
			int[] enclosingRect = crater.enclosingRect;
			Point upperLeftCorner = new Point(enclosingRect[0], enclosingRect[1]);
			Point lowerRightCorner = new Point(enclosingRect[2], enclosingRect[3]);
			//	Update color based on radius
			Core.rectangle(matResult, upperLeftCorner, lowerRightCorner, new Scalar(255,0,0));
		}

		BufferedImage resultImg = Util.Mat2BufferedImage(matResult);
		resultImgPanel.setImage(resultImg);
		resultImgPanel.repaint();
	}

	public static class CustomPanel extends JPanel
	{
		private static final long serialVersionUID = 1L;
		private BufferedImage image;

		public CustomPanel()
		{

		}

		public CustomPanel(BufferedImage image)
		{
			this.image = image;
		}

		public void paintComponent(Graphics g)
		{
			super.paintComponent(g);
			g.drawImage(image, 0, 0, this);
		}

		public void setImage(BufferedImage image)
		{
			this.image = image;
		}
	}

	/*
	public static void writeData(ArrayList<Crater> accumCircles, int rows, int cols)
	{
		try {
			File file = new File("crater_data.txt");
			BufferedWriter output = new BufferedWriter(new FileWriter(file));

			output.write("Rows: "+rows+"\n");
			output.write("Columns: "+cols+"\n");
			output.write("\n");

			int id = 1;
			double mapX = 0.0;
			double mapY = 0.0;
			double latitude = 0.0;
			double longitude = 0.0;
			double majorAx = 0.0;
			double minorAx = 0.0;
			double orientation = 0.0;
			double depth = 0.0;

			//String format = "%-15s %-15s %-15s %-15s %-15s %-15s %-15s %-15s %-15s %-15s %-15s %-15s %-15s %-15s %-15s%n";
			String format = "%-15s %-15s %-15s %-15s %-15s%n";
			//output.write(String.format(format, "Id", "NACx", "NACy", "Box Size", "MapX", "MapY", "Latitude", "Longitude", "MajorAx", "MinorAx", "Orientation", "Diameter", "Depth", "UpperLeft", "LowerRight"));
			output.write(String.format(format, "Id", "NACx", "NACy", "UpperLeft", "LowerRight"));
			for (int i=0; i<accumCircles.size(); i++)
			{
				Crater crater = accumCircles.get(i);
				double cx = crater.centerX;
				double cy = crater.centerY;
				double r = crater.radius;
				double boxSize = 0.0;
				int[] enclosingRect = crater.enclosingRect;
				//output.write(String.format(format, id, cx, cy, boxSize, mapX, mapY, latitude, longitude, majorAx, minorAx, orientation, r*2.0, depth, "("+enclosingRect[0]+","+enclosingRect[1]+")", "("+enclosingRect[2]+","+enclosingRect[3]+")"));
				output.write(String.format(format, id, cx, cy, "("+enclosingRect[0]+","+enclosingRect[1]+")", "("+enclosingRect[2]+","+enclosingRect[3]+")"));
				id++;
			}
			output.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	*/
}
