package lipfd.dataExtract;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferByte;
import java.io.IOException;
import java.util.*;
import java.util.List;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.MouseInputListener;

import org.opencv.core.Core;
import org.opencv.core.Mat;
import org.opencv.core.Point;
import org.opencv.core.Rect;
import org.opencv.core.Scalar;

import lipfd.commons.Crater;
import lipfd.commons.Image;
import lipfd.commons.Util;
import lipfd.commons.model.ImageMetadata;
import lipfd.commons.model.dao.ImageMetadataDao;
import lipfd.commons.model.dao.jdbc.ImageMetadataDaoImpl;
import lipfd.performanceCalculator.PerformanceCalculator;

public class DataExtractGUI implements ActionListener {

	private BufferedImage image = new BufferedImage(10, 10,
			BufferedImage.TYPE_INT_RGB);
	private Mat imgMat;

	// GUI variables
	private ImagePanel imgPanel;
	private JScrollPane imgPane;
	private JLabel craterListLabel = new JLabel("Craters List");
	private JPanel inputFieldsPanel;
	private JTextField jtfLatitude = new JTextField(5);
	private JTextField jtfLongitude = new JTextField(5);
	private JTextField jtfSunAngle = new JTextField(5);
	private JTextField jtfAzimuth = new JTextField(5);
	private JTextField jtfResolution = new JTextField(5);
	// test fields to display depth/diameter
	private JTextField tComputedDepth = new JTextField(10);
	private JTextField tComputedDiameter = new JTextField(10);
	// Menu options
	private JMenuItem drawShadow;
	private JMenuItem drawDiameter;
	private JMenuItem mExit;
	// List of all craters for user selection.
	private JList<String> cratersSelectionList = new JList<String>();
	// Index of the currently selected crater.
	private int craterIndex = 0;
	// Current tool selected.
	private int currentTool = 0;
	// User specified shadow length endpoints.
	private Point[] points = new Point[4];
	// Do not display crater list yet until after computation of the two images
	// of the crater
	private boolean alreadyComputed = false;
	// Initialize craters image, metadata, and list.
	private Image cratersImage;
	// no need to switch metadata classes since we're only using one now
	private ImageMetadata metadata; // = new ImageMetadata();
	private List<Crater> craterList = new ArrayList<Crater>();

	private JFrame frame;

	public static void main(String args[]) {
		System.loadLibrary(Core.NATIVE_LIBRARY_NAME);

		String productID = args[0];
		int ulx = Integer.parseInt(args[1]);
		int uly = Integer.parseInt(args[2]);
		int lrx = Integer.parseInt(args[3]);
		int lry = Integer.parseInt(args[4]);

		double confThresh = Double.parseDouble(args[5]);

		// Code to download image or pull image from file if it already exists.
		ImageMetadataDao imageDao = new ImageMetadataDaoImpl("lipfd", "lipfd");
		// use bestInputImageMetadata for getting metadata or the image
		ImageMetadata bestInputImageMetadata = imageDao.getMetadata(productID);
		System.out.println("\nchecking the downloaded files.");
		if (!Util.fileExists("downloadedImages/" + productID + ".IMG")) {
			System.out.println("downloading the image:");
			Util.download("http://lroc.sese.asu.edu/data/"
					+ bestInputImageMetadata.file_specification_name,
					"downloadedImages/" + productID + ".IMG");
		}

		String originalInputImagePath = "downloadedImages/" + productID
				+ ".IMG";
		String newInputImagePath = "";
		String commandString = "";
		if (System.getProperty("os.name").equalsIgnoreCase("linux")) {
			newInputImagePath = "downloadedImages/" + productID + ".pgm";
			commandString = "gdal_translate -srcwin 0 0 "
					+ String.valueOf(bestInputImageMetadata.line_samples) + " "
					+ String.valueOf(bestInputImageMetadata.image_lines)
					+ " -ot Byte -of PNM -scale " + originalInputImagePath
					+ " " + newInputImagePath;
		} else {
			newInputImagePath = "downloadedImages/"
					+ bestInputImageMetadata.product_id + ".tiff";
			commandString = "gdal_translate -srcwin 0 0 "
					+ String.valueOf(bestInputImageMetadata.line_samples) + " "
					+ String.valueOf(bestInputImageMetadata.image_lines) + " "
					+ originalInputImagePath + " " + newInputImagePath
					+ " -ot Byte -scale 0 4095 0 255";
		}
		System.out.println("running gdal:");
		System.out.println(commandString);
		if (!Util.fileExists(newInputImagePath))
			Util.runProgram(commandString);
		Image inputImage = new Image(newInputImagePath)
				.crop(ulx, uly, lrx, lry);
		Mat clone = inputImage.getMat().clone();
		Image cloneI = new Image(clone);
		// /end code for getting image and metadata//

		List<Crater> polishedCraterList = new ArrayList<Crater>();
		String metadataString = "";
		try {
			metadataString = Util.readFile("results/result-ml-metadata.txt");
		} catch (IOException e) {
			e.printStackTrace();
		}
		polishedCraterList = Util.parseMetaData(metadataString, (int) ulx,
				(int) uly, (int) lrx, (int) lry);
		for(Crater c : polishedCraterList){
			c.addMargin(inputImage.getWidth(), inputImage.getHeight(), 0.5);
		}
		// update shadow values for each crater
		List<Image> images = new ArrayList<Image>();
		images.add(inputImage);
		List<ImageMetadata> metadatas = new ArrayList<ImageMetadata>();
		metadatas.add(bestInputImageMetadata);
		// set depth and diameter for each crater
		DataExtract.computeDDs(polishedCraterList, images, metadatas,
				bestInputImageMetadata.sub_solar_azimuth, confThresh, true);
		List<Crater> filteredList = new ArrayList<Crater>();
		// create a list of craters with confidence values greater than
		// threshvalue.
		for (Crater c : polishedCraterList) {
			if (c.conf >= confThresh) {
				filteredList.add(c);
			}
		}
		DataExtractGUI rtGUI = new DataExtractGUI((ArrayList<Crater>)filteredList, cloneI, bestInputImageMetadata);

	}// end main

	public DataExtractGUI(List<Crater> craterList, Image cratersImage,
			ImageMetadata metadata) {
		this.craterList = craterList;
		// Initialized user specified shadow length endpoints.
		for (int i = 0; i < points.length; i++) {
			points[i] = new Point(0, 0);
		}
		this.cratersImage = cratersImage;

		// Initialized to first crater image and used its metadata.
		imgMat = cratersImage.getMat().clone();
		List<List<Point>> endPoints;
		endPoints = craterList.get(craterIndex).shadowEndPoints;
		//Core.line(imgMat, endPoints.get(0).get(0), endPoints.get(0).get(1), new Scalar(255,0,255));
		image = matToBufferedImage(imgMat);
		this.metadata = metadata;
		tComputedDiameter.setText(String.format("%.2f", craterList.get(craterIndex).diameter));
		tComputedDepth.setText(String.format("%.2f", craterList.get(craterIndex).depth));

		displayImageGUI(metadata);
	}

	public void displayImageGUI(ImageMetadata md) {

		frame = new JFrame();
		JPanel craterListPanel = new JPanel();

		craterListPanel.setLayout(new BorderLayout());

		JMenuBar menu = new JMenuBar();
		JMenu fileMenu = new JMenu("File");

		// Tools menu.
		JMenu toolsMenu = new JMenu("Tools");
		drawShadow = new JMenuItem("Draw Shadow");
		drawShadow.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				currentTool = 1;
			}
		});
		drawShadow.addActionListener(this);

		drawDiameter = new JMenuItem("Draw Diameter");
		drawDiameter.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				currentTool = 2;
			}
		});
		drawDiameter.addActionListener(this);

		toolsMenu.add(drawShadow);
		toolsMenu.add(drawDiameter);

		mExit = new JMenuItem("Exit");
		mExit.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				System.exit(0);
			}
		});
		fileMenu.add(mExit);
		menu.add(fileMenu);
		menu.add(toolsMenu);

		String[] craterSelection = new String[craterList.size()];
		if (alreadyComputed) {
			int i = 1;
			while ((i - 1) < craterList.size()) {
				craterSelection[i - 1] = ("Crater " + i);
				i++;
			}
		}
		cratersSelectionList = new JList<String>(craterSelection);

		ListSelectionModel listSelectionModel = cratersSelectionList
				.getSelectionModel();
		listSelectionModel.addListSelectionListener(new CraterListListener());
		cratersSelectionList
				.setSelectionMode(ListSelectionModel.SINGLE_INTERVAL_SELECTION);
		cratersSelectionList.setLayoutOrientation(JList.VERTICAL);
		cratersSelectionList.setVisibleRowCount(-1);
		JScrollPane listPane = new JScrollPane(cratersSelectionList);

		listPane.setPreferredSize(new Dimension(100, 80));

		final Dimension dimensionSize = new Dimension(image.getWidth(),
				image.getHeight());
		imgPanel = new ImagePanel();
		imgPanel.setPreferredSize(dimensionSize);
		imgPane = new JScrollPane(imgPanel);

		// test box for average pixel width/height
		double resolution = (metadata.scaled_pixel_width*metadata.image_lines+metadata.scaled_pixel_width)/2.0;
		//double resolution = (md.scaled_pixel_height + md.scaled_pixel_height) / 2.0;
		jtfResolution.setText(String.format("%.2f", resolution));

		// test box for latitude
		jtfLatitude.setText(md.center_latitude + "");

		// text box for longitude
		jtfLongitude.setText(md.center_longitude + "");

		// Text box for sun angle
		jtfSunAngle.setText(md.incidence_angle + "");

		// text box for azimuth
		jtfAzimuth.setText(md.sub_solar_azimuth + "");

		JButton cButton = new JButton("Compute");
		cButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				alreadyComputed = true;
				frame.dispose();
				displayImageGUI(md);
			}
		});

		// Recompute the crater's depth with the user specified shadow length.
		JButton updateButton = new JButton("Update");
		updateButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if((points[0].x != 0 && points[0].y != 0) && (points[1].x != 0 && points[1].y != 0)){
					DataExtract.recomputeDepthSI(craterList.get(craterIndex),
						metadata, points[0], points[1]);
				}
				if((points[2].x != 0 && points[2].y != 0) && (points[3].x != 0 && points[3].y != 0)){
					DataExtract.recomputeDiameterSI(craterList.get(craterIndex),
						metadata, points[2], points[3]);
				}
				tComputedDiameter.setText(String.format("%.2f", craterList.get(craterIndex).diameter));
				tComputedDepth.setText(String.format("%.2f", craterList.get(craterIndex).depth));
			}
		});

		JPanel southPanel = new JPanel();
		JPanel depthDiameterP = new JPanel();
		depthDiameterP.add(new JLabel("Diameter: "));
		depthDiameterP.add(tComputedDiameter);
		depthDiameterP.add(new JLabel("Depth: "));
		depthDiameterP.add(tComputedDepth);

		inputFieldsPanel = new JPanel();
		inputFieldsPanel.setLayout(new FlowLayout());
		inputFieldsPanel.add(new JLabel("Res: "));
		inputFieldsPanel.add(jtfResolution);
//		inputFieldsPanel.add(new JLabel("Longitude: "));
//		inputFieldsPanel.add(jtfLongitude);
//		inputFieldsPanel.add(new JLabel("Latitude: "));
//		inputFieldsPanel.add(jtfLatitude);
		inputFieldsPanel.add(new JLabel("Sun Angle: "));
		inputFieldsPanel.add(jtfSunAngle);
		inputFieldsPanel.add(new JLabel("Azimuth Angle: "));
		inputFieldsPanel.add(jtfAzimuth);
		inputFieldsPanel.add(cButton);
		inputFieldsPanel.add(updateButton);

		craterListPanel.add(craterListLabel, BorderLayout.NORTH);
		craterListPanel.add(listPane, BorderLayout.CENTER);

		southPanel.add(depthDiameterP, BorderLayout.NORTH);
		southPanel.add(inputFieldsPanel, BorderLayout.SOUTH);

		frame.setJMenuBar(menu);
		frame.add(imgPane, BorderLayout.CENTER);
		frame.add(craterListPanel, BorderLayout.EAST);
		//frame.add(inputFieldsPanel, BorderLayout.SOUTH);
		frame.add(southPanel, BorderLayout.SOUTH);
		frame.pack();
		if (frame.getHeight() > 500) {
			frame.setSize(frame.getWidth(), 500);
		}
		frame.setTitle("Ringtoss");
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.setVisible(true);
	}

	class CraterListListener implements ListSelectionListener {

		public void valueChanged(ListSelectionEvent e) {
			// Retrieve the list selection model and figure out what crater
			// number was selected. The crater number
			// is one less than the crater number shown on screen.
			ListSelectionModel lsm = (ListSelectionModel) e.getSource();
			int minIndex = lsm.getMinSelectionIndex();
			int maxIndex = lsm.getMaxSelectionIndex();
			for (int i = minIndex; i <= maxIndex; i++) {
				if (lsm.isSelectedIndex(i)) {
					craterIndex = i;
				}
			}

			// Change image and metadata when a different crater is selected.
			imgMat = cratersImage.getMat().clone();
			List<List<Point>> endPoints;
			endPoints = craterList.get(craterIndex).shadowEndPoints;
			Point upperBound = new Point(craterList.get(craterIndex).enclosingRect[0], craterList.get(craterIndex).enclosingRect[1]);
			Point lowerBound = new Point(craterList.get(craterIndex).enclosingRect[2], craterList.get(craterIndex).enclosingRect[3]);
			double startX = endPoints.get(0).get(0).x + upperBound.x;
			double startY = endPoints.get(0).get(0).y + upperBound.y;
			double endX = endPoints.get(0).get(1).x + upperBound.x;
			double endY = endPoints.get(0).get(1).y + upperBound.y;
			Point startPoint = new Point(startX, startY);
			Point endPoint = new Point(endX, endY);

			Core.rectangle(imgMat, upperBound, lowerBound, new Scalar(255, 255, 255), 2);
			Core.line(imgMat, startPoint, endPoint, new Scalar(255,0,255));
			image = matToBufferedImage(imgMat);
			tComputedDiameter.setText(String.format("%.2f", craterList.get(craterIndex).diameter));
			tComputedDepth.setText(String.format("%.2f", craterList.get(craterIndex).depth));

			// Reset points for new crater image.
			for (int i = 0; i < points.length; i++) {
				points[i].x = points[i].y = 0;
			}

			frame.repaint();
		}
	}

	public class ImagePanel extends JPanel implements MouseInputListener,
			MouseListener, MouseMotionListener {
		private static final long serialVersionUID = 1L;

		// Use to keep track of user mouse clicks when drawing shadow length
		// line.
		private int count = -1;
		private int dcount = -1;

		public ImagePanel() {

			addMouseListener(this);
			addMouseMotionListener(this);

		}

		public void paintComponent(Graphics pen) {
			super.paintComponent(pen);
			pen.drawImage(image, 0, 0, this);

			paintDiameters(pen);

			for (int i = 0; i < 4; i++) {

				// Different colors for each tool
				if (i == 0) {
					pen.setColor(Color.RED);
				} else {
					pen.setColor(Color.GREEN);
				}

				int tempCount = count;
				if (i == 2) {
					tempCount = dcount;
				}
				if (i % 2 == 0) {
					if (tempCount == 1) {
						if (points[i + 1].x != 0.0) {
							pen.fillRect((int) points[i].x, (int) points[i].y,
									5, 5);
							pen.fillRect((int) points[i + 1].x,
									(int) points[i + 1].y, 5, 5);
							pen.drawLine((int) points[i].x, (int) points[i].y,
									(int) points[i + 1].x,
									(int) points[i + 1].y);

						} else {
							pen.fillRect((int) points[i].x, (int) points[i].y,
									5, 5);
						}
					} else if (tempCount == 2) {
						pen.fillRect((int) points[i].x, (int) points[i].y, 5, 5);
						pen.fillRect((int) points[i + 1].x,
								(int) points[i + 1].y, 5, 5);
						pen.drawLine((int) points[i].x, (int) points[i].y,
								(int) points[i + 1].x, (int) points[i + 1].y);
						if (i == 2) {
							dcount = 0;
						} else {
							count = 0;
						}
					} else if (tempCount == 0) {
						pen.fillRect((int) points[i].x, (int) points[i].y, 5, 5);
						pen.fillRect((int) points[i + 1].x,
								(int) points[i + 1].y, 5, 5);
						pen.drawLine((int) points[i].x, (int) points[i].y,
								(int) points[i + 1].x, (int) points[i + 1].y);
					}
				}
			}
		}

		// Draw diameters of craters in the image
		public void paintDiameters(Graphics pen) {
			super.paintComponent(pen);
			pen.drawImage(image, 0, 0, this);
			int displayIndex = craterIndex + 1;
			int centerX = (int) craterList.get(craterIndex).centerX - ((int)craterList.get(craterIndex).radius + 5);
			int centerY = (int) craterList.get(craterIndex).centerY - ((int)craterList.get(craterIndex).radius + 5);
			pen.setColor(Color.YELLOW);
			pen.setFont(new Font(pen.getFont().getFontName(), Font.BOLD, 10));
			pen.setColor(Color.YELLOW);
			pen.drawString("#" + displayIndex, centerX, centerY);

		}

		public void mouseClicked(MouseEvent e) {

			// Draw shadow length of a crater manually.
			if (currentTool == 1) {
				if (count == -1) {
					count = 1;
				} else {
					count++;
				}

				// First mouse click.
				if (count == 1) {
					points[1].x = points[1].y = 0;
					int x = e.getX();
					int y = e.getY();
					points[0].x = x;
					points[0].y = y;
				}

				// Second mouse click.
				else if (count == 2) {
					int x = e.getX();
					int y = e.getY();
					points[1].x = x;
					points[1].y = y;

					List<Point> newPoints = new ArrayList<Point>();
					newPoints.add(new Point(points[0].x, points[0].y));
					newPoints.add(new Point(points[1].x, points[1].y));
					craterList.get(craterIndex).shadowEndPoints.add(newPoints);
				}
				repaint();
			}

			// Draw diameter length of a crater manually.
			else {
				if (dcount == -1) {
					dcount = 1;
				} else {
					dcount++;
				}

				// First mouse click.
				if (dcount == 1) {
					points[3].x = points[3].y = 0;
					int x = e.getX();
					int y = e.getY();
					points[2].x = x;
					points[2].y = y;
				}

				// Second mouse click.
				else if (dcount == 2) {
					int x = e.getX();
					int y = e.getY();
					points[3].x = x;
					points[3].y = y;
				}
				repaint();
			}
		}

		public void mouseEntered(MouseEvent arg0) {

		}

		public void mouseExited(MouseEvent arg0) {

		}

		public void mousePressed(MouseEvent e) {

		}

		public void mouseReleased(MouseEvent e) {

		}

		public void mouseDragged(MouseEvent e) {

		}

		public void mouseMoved(MouseEvent e) {

		}
	}

	// Convert matrix of the crater image into a buffered image.
	public static BufferedImage matToBufferedImage(Mat m) {
		int type = BufferedImage.TYPE_BYTE_GRAY;
		if (m.channels() > 1) {
			type = BufferedImage.TYPE_3BYTE_BGR;
		}
		int bufferSize = m.channels() * m.cols() * m.rows();
		byte[] b = new byte[bufferSize];
		m.get(0, 0, b);
		BufferedImage img = new BufferedImage(m.cols(), m.rows(), type);
		final byte[] targetPixels = ((DataBufferByte) img.getRaster()
				.getDataBuffer()).getData();
		System.arraycopy(b, 0, targetPixels, 0, b.length);

		return img;
	}

	@Override
	public void actionPerformed(ActionEvent e) {
		// TODO Auto-generated method stub

	}
}
