package lipfd.commons;

import org.opencv.core.Core;
import org.opencv.core.CvType;
import org.opencv.core.Mat;
import org.opencv.core.Scalar;
import org.opencv.highgui.*;
import org.opencv.imgproc.Imgproc;
import org.opencv.core.Rect;
import org.opencv.core.Size;
import org.opencv.core.RotatedRect;
import org.opencv.core.Point;

import java.util.*;



public class Image {
	private Mat mat;
	private String filename;

	public Image(Mat img){
		mat = img;
		filename = null;
	}
	public Image(String filename){
		mat = Highgui.imread(filename, Highgui.CV_LOAD_IMAGE_GRAYSCALE);
		this.filename = filename;
	}
	public Mat getMat(){
		return mat;
	}
	public int getWidth(){
		return mat.width();
	}
	public int getHeight(){
		return mat.height();
	}
	public int getArea(){
		return mat.width() * mat.height();
	}
	public int getChannels(){
		return mat.channels();
	}
	public int getDepth(){
		return mat.depth();
	}
	public String getFilename(){
		return filename;
	}
	public int[] getPixels(){
		byte[] pixelsByte = new byte[getWidth() * getHeight()];
		int size = mat.get(0,0,pixelsByte);
		int[] pixels = new int[size];
		for(int i = 0; i < size; i++){
			pixels[i] = pixelsByte[i] & 0xFF;
		}
		return pixels;
	}
	public boolean saveImage(){
		if(filename != null)
			return Highgui.imwrite(filename, mat);
		return false;
	}
	public boolean saveImage(String filename){
		this.filename = filename;
		return Highgui.imwrite(filename, mat);
	}
	public Core.MinMaxLocResult minMaxLoc(){
		return Core.minMaxLoc(mat);
	}
	public Image equalizeHistogram(){
		Mat equalized = new Mat();
		Imgproc.equalizeHist(mat, equalized);
		return new Image(equalized);
	}
	public Image medianFilter(int k){
		Mat blurredImage = new Mat();
		Imgproc.medianBlur(mat, blurredImage, k);
		return new Image(blurredImage);
	}
	public Image invert(){
		Mat invertedImage = new Mat();
		Core.bitwise_not(mat, invertedImage);
		return new Image(invertedImage);
	}
	public Image resize(double width, double height){
		Mat resized = new Mat();
		Imgproc.resize(mat, resized, new Size(width, height), 0, 0,
			 (mat.width()*mat.height() > width*height)?Imgproc.INTER_AREA:Imgproc.INTER_CUBIC);
		return new Image(resized);
	}
	public Image crop(int minx, int miny, int maxx, int maxy){
		minx = Math.max(minx, 0);
		miny = Math.max(miny, 0);
		maxx = Math.min(mat.width(), maxx);
		maxy = Math.min(mat.height(), maxy);
		Mat cropped = new Mat(mat, new Rect(minx, miny, maxx - minx, maxy - miny));
		return new Image(cropped);
	}
	public Image crop(double minx, double miny, double maxx, double maxy){
		return crop((int)minx, (int)miny, (int)maxx, (int)maxy);
	}
	public Image cropRotated(Point center, Size size, double angle, boolean flipAroundHorizontalAxis, boolean flipAroundVerticalAxis){
		RotatedRect rect = new RotatedRect(center, size, angle);
		Rect boundingRect = rect.boundingRect();
		Mat cropped = this.crop(boundingRect.x, boundingRect.y, boundingRect.x + boundingRect.width, boundingRect.y + boundingRect.height).getMat();
		Mat src = new Mat();
		Point centerRect = null;
		if(flipAroundHorizontalAxis && flipAroundVerticalAxis){
			Core.flip(cropped, src, -1);
			centerRect = new Point(boundingRect.width - center.x + boundingRect.x,
				boundingRect.height - center.y + boundingRect.y);
		}
		else if(flipAroundVerticalAxis){
			Core.flip(cropped, src, 1);
			centerRect = new Point(boundingRect.width - center.x + boundingRect.x,
				center.y - boundingRect.y);
		}
		else if(flipAroundHorizontalAxis){
			Core.flip(cropped, src, 0);
			centerRect = new Point(center.x - boundingRect.x,
				boundingRect.height - center.y + boundingRect.y);
		}
		else {
			src = cropped;
			centerRect = new Point(center.x - boundingRect.x, center.y - boundingRect.y);
		}
		Mat rotated = new Mat();
		Mat m = Imgproc.getRotationMatrix2D(centerRect, angle, 1.0);
		Imgproc.warpAffine(src, rotated, m, src.size(), Imgproc.INTER_CUBIC);
		Mat result = new Mat();
		Imgproc.getRectSubPix(rotated, rect.size, centerRect, result);
		return new Image(result);
	}
	public Image flip(boolean flipAroundHorizontalAxis, boolean flipAroundVerticalAxis){
		Mat result = new Mat();
		if(flipAroundHorizontalAxis && flipAroundVerticalAxis){
			Core.flip(mat, result, -1);
		}
		else if(flipAroundVerticalAxis){
			Core.flip(mat, result, 1);
		}
		else if(flipAroundHorizontalAxis){
			Core.flip(mat, result, 0);
		}
		else {
			result = mat;
		}
		return new Image(result);
	}
	public Image aboveLevel(int level){
		int[] pixels = getPixels();
		byte[] newPixels = new byte[pixels.length];
		for(int i = 0; i < pixels.length; i++){
			if(pixels[i] >= level)
				newPixels[i] = (byte) 0xFF;
			else newPixels[i] = (byte) 0x00;
		}
		Mat newMat = new Mat(getHeight(), getWidth(), CvType.CV_8UC1);
		newMat.put(0, 0, newPixels);
		return new Image(newMat);
	}
	public Image belowLevel(int level){
		int[] pixels = getPixels();
		byte[] newPixels = new byte[pixels.length];
		for(int i = 0; i < pixels.length; i++){
			if(pixels[i] <= level)
				newPixels[i] = (byte) 0xFF;
			else newPixels[i] = (byte) 0x00;
		}
		Mat newMat = new Mat(getHeight(), getWidth(), CvType.CV_8UC1);
		newMat.put(0, 0, newPixels);
		return new Image(newMat);
	}
}
